/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.ui.components.ds;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceMetadata;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.SyntheticResource;
import org.apache.sling.api.resource.ValueMap;

import java.util.Collection;
import java.util.Iterator;

/**
 * An implementation of resource that will return the ValueMap passed, when this
 * resource is adapted to ValueMap.
 */
public class ValueMapResource extends SyntheticResource {
    private ValueMap vm;
    private Collection<Resource> children;

    /**
     * Creates a <code>ValueMapResource</code> based on provided parameters
     * @param resourceResolver The resource resolver
     * @param path The resource path
     * @param resourceType The resource type
     * @param vm The properties of the resource
     */
    public ValueMapResource(ResourceResolver resourceResolver, String path, String resourceType, ValueMap vm) {
        super(resourceResolver, path, resourceType);
        this.vm = vm;
    }

    /**
     * Creates a <code>ValueMapResource</code> based on provided parameters, with the ability to define the children
     * @param resourceResolver The resource resolver
     * @param path The resource path
     * @param resourceType The resource type
     * @param vm The properties of the resource
     * @param children The children of the resource
     */
    public ValueMapResource(ResourceResolver resourceResolver, String path, String resourceType, ValueMap vm, Collection<Resource> children) {
        this(resourceResolver, path, resourceType, vm);
        this.children = children;
    }

    /**
     * Creates a <code>ValueMapResource</code> based on provided parameters
     * @param resourceResolver The resource resolver
     * @param rm The resource meta data
     * @param resourceType The resource type
     * @param vm The properties of the resource
     */
    public ValueMapResource(ResourceResolver resourceResolver, ResourceMetadata rm, String resourceType, ValueMap vm) {
        super(resourceResolver, rm, resourceType);
        this.vm = vm;
    }

    /**
     * Creates a <code>ValueMapResource</code> based on provided parameters, with the ability to define the children
     * @param resourceResolver The resource resolver
     * @param rm The resource meta data
     * @param resourceType The resource type
     * @param vm The properties of the resource
     * @param children The children of the resource
     */
    public ValueMapResource(ResourceResolver resourceResolver, ResourceMetadata rm, String resourceType, ValueMap vm, Collection<Resource> children) {
        this(resourceResolver, rm, resourceType, vm);
        this.children = children;
    }

    @SuppressWarnings("unchecked")
    @Override
    public <Type> Type adaptTo(Class<Type> type) {
        if (ValueMap.class.equals(type)) {
            return (Type) vm;
        } else {
            return super.adaptTo(type);
        }
    }

    /**
     * @see org.apache.sling.api.resource.Resource#getChild(String)
     */
    @Override
    public Resource getChild(String relPath) {
        if (children == null) {
            return super.getChild(relPath);
        }
        for (Resource child : children) {
            if (child.getPath() != null && child.getPath().equals(getPath() + "/" + relPath)) {
                return child;
            }
        }
        return null;
    }

    /**
     * @see org.apache.sling.api.resource.Resource#listChildren()
     */
    @Override
    public Iterator<Resource> listChildren() {
        if (children == null) {
            return super.listChildren();
        }
        return children.iterator();
    }

    /**
     * @see org.apache.sling.api.resource.Resource#getChildren()
     */
    @Override
    public Iterable<Resource> getChildren() {
        return new Iterable<Resource>() {
            public Iterator<Resource> iterator() {
                return listChildren();
            }
        };
    }

    /**
     * @see org.apache.sling.api.resource.Resource#hasChildren()
     */
    @Override
    public boolean hasChildren() {
        if (children == null) {
            return super.hasChildren();
        }
        return !children.isEmpty();
    }

}
