/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.workflow.launcher;

import java.util.List;


/**
 * The <code>ConfigEntry</code> represents <i>one</i> workflow launcher 
 * configuration entry
 */
public class ConfigEntry {
    private String nodetype;
    private String whereClause;
    private int eventType;
    private String glob;
    private String workflow;
    private String id;
    private String description;
    private boolean enabled;
    private List<String> excludeList;
    private List<String> runModes;

    /**
     * Construct a ConfigEntry with the specified information.
     * @param eventType one of javax.jcr.observation.Event#NODE_ADDED
     * @param glob the path pattern, can be regexp, where the event should be watching
     * @param nodetype type of node to watch
     * @param whereClause node property conditions to watch
     * @param workflow the name of the workflow to trigger
     * @param id the id for this config
     * @param description description for this config
     * @param enabled true if the enabled, false for disabled
     * @param excludeList conditions to exclude from the event
     * @param runModes runmodes for this config
     */
    public ConfigEntry(int eventType, String glob, String nodetype,
                       String whereClause, String workflow, String id,
                       String description, boolean enabled,
                       List<String> excludeList, List<String> runModes) {
        this.eventType = eventType;
        this.glob = glob;
        this.nodetype = nodetype;
        this.whereClause = whereClause;
        this.workflow = workflow;
        this.id = id;
        this.description = description;
        this.enabled = enabled;
        this.excludeList = excludeList;
        this.runModes = runModes;
    }

    /**
     * Return the name of the workflow to trigger.
     * @return workflow name.
     */
    public String getWorkflow() {
        return workflow;
    }

    /**
     * Set the name of the workflow to trigger.
     * @param workflow workflow name
     */
    public void setWorkflow(String workflow) {
        this.workflow = workflow;
    }

    /**
     * Return the JCR Event Type that triggers this config.
     * @return event type
     */
    public int getEventType() {
        return eventType;
    }

    /**
     * Set the event type to watch.
     * @param eventType event type to listen to.
     */
    public void setEventType(int eventType) {
        this.eventType = eventType;
    }

    /**
     * Returns the path pattern that triggers this launcher configuration.
     * @return path pattern.
     */
    public String getGlob() {
        return glob;
    }

    /**
     * Set the path pattern to trigger this launcher configuration.
     * @param glob path pattern.
     */
    public void setGlob(String glob) {
        this.glob = glob;
    }

    /**
     * Get the node type associated with this launcher.
     * @return node type
     */
    public String getNodetype() {
        return nodetype;
    }

    /**
     * Set the node type associated with this launcher.
     * @param nodetype the node type
     */
    public void setNodetype(String nodetype) {
        this.nodetype = nodetype;
    }

    /**
     * Return the where clause for this trigger.
     * The where clause is a condition of node properties,
     * e.g. approved==true
     *
     * @return where clause
     */
    public String getWhereClause() {
        return whereClause;
    }

    /**
     * Set the where clause for this trigger.
     * @param whereClause condition
     */
    public void setWhereClause(String whereClause) {
        this.whereClause = whereClause;
    }

    /**
     * Return the ID of this trigger.
     * @return id
     */
    public String getId() {
        return id;
    }

    /**
     * Set the Id of this trigger
     * @param id trigger id.
     */
    public void setId(String id) {
        this.id = id;
    }

    /**
     * Get the description for this launcher configuration.
     * @return description
     */
    public String getDescription() {
        return description;
    }

    /**
     * Set the description for this launcher configuration.
     * @param description a new description.
     */
    public void setDescription(String description) {
        this.description = description;
    }

    public int hashCode() {
        String code = String.valueOf(eventType) + nodetype + whereClause + glob + workflow;
        return code.hashCode();
    }


    /**
     * Returns if this launcher configuration is enabled or not.
     * @return true if the configuration is enabled, false otherwise.
     */
    public boolean isEnabled() {
        return enabled;
    }

    /**
     * Enables or disables this launcher configuration.
     * @param enabled true to enable this configuration, false otherwise.
     */
    public void setEnabled(boolean enabled) {
        this.enabled = enabled;
    }

    /**
     * Returns a list of exclude conditions.
     * If conditions in this list evaluate to true the launcher configuration does not trigger.
     * @return list of conditions
     */
    public List<String> getExcludeList() {
        return excludeList;
    }

    /**
     * Set the list of exclude conditions.
     * @param excludeList new exclude conditions list.
     */
    public void setExcludeList(List<String> excludeList) {
        this.excludeList = excludeList;
    }

    /**
     * Return the list of runmode associated with this launcher configuration.
     * @return list of run modes.
     */
    public List<String> getRunModes() {
        return this.runModes;
    }

    /**
     * Set the run moodes associated with this launcher configuration.
     * @param runModes list of runmodes.
     */
    public void setRunModes(List<String> runModes) {
        this.runModes = runModes;
    }

    @Override
    public String toString() {
        return "ConfigEntry{" +
                "nodetype='" + nodetype + '\'' +
                ", whereClause='" + whereClause + '\'' +
                ", eventType=" + eventType +
                ", glob='" + glob + '\'' +
                ", workflow='" + workflow + '\'' +
                ", id='" + id + '\'' +
                ", description='" + description + '\'' +
                ", enabled=" + enabled +
                ", excludeList=" + excludeList +
                ", runModes=" + runModes +
                '}';
    }
}
