/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2015 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.workflow.payload;

import java.util.Calendar;
import java.util.Map;

import aQute.bnd.annotation.ProviderType;

/**
 *  PayloadInfo is used to store all the available information about an {@link }InboxItem}'s payload.  The most
 *  needed information is represented as first class properties.  Any other information needed can be added to the
 *  custom properties Map.
 */
@ProviderType
public interface PayloadInfo {

    /** Known Payload types **/
    public enum PAYLOAD_TYPE {
        PAGE, ASSET, PROJECT, URL, JCR_PATH
    }

    /**
     * Return the path of the payload content.
     * @return the path to the payload content
     * @see InboxItem.getContentPath
     */
    String getPath();

    /**
     * Set the path of the payload content.
     * @param path the path to the payload content
     * @return this PayloadInfo object
     * @see InboxItem.getContentPath
     */
    PayloadInfo setPath(String path);

    /**
     * Return the path to be opened in a browser to load the representation of the payload.
     * @return the path to the payload to be opened in a browser
     */
    String getBrowserPath();

    /**
     * Set the path to be opened in a browser to load the representation of the payload.
     * @param path the path to the payload to be opened in a browser
     * @return this PayloadInfo object
     */
    PayloadInfo setBrowserPath(String browserPath);

    /**
     * Return the title associated with the payload.
     * @return the payload's title
     */
    String getTitle();

    /**
     * Set the title associated with the payload.
     * @param title the payload's title
     * @return this PayloadInfo object
     */
    PayloadInfo setTitle(String title);

    /**
     * Return the path to the thumbnail that represents the payload
     * @return the payload's thumbnail path
     */
    String getThumbnailPath();

    /**
     * Set the path to the thumbnail that represents the payload
     * @param thumbnailPath the payload's thumbnail path
     * @return this PayloadInfo object
     */
    PayloadInfo setThumbnailPath(String thumbnailPath);

    /**
     * Return the type of the payload
     * @return the payload's type
     * @see com.adobe.granite.workflow.payload.PayloadInfo.PAYLOAD_TYPE
     */
    String getPayloadType();

    /**
     * Set the type of the payload
     * @param payloadType the payload's type
     * @return this PayloadInfo objet
     * @see com.adobe.granite.workflow.payload.PayloadInfo.PAYLOAD_TYPE
     */
    PayloadInfo setPayloadType(String payloadType);

    /**
     * Return the date and time when the payload was created.
     * @return the {@link Calendar} representing when the payload was created
     */
    Calendar getCreated();

    /**
     * Set the date and time when the payload was created.
     * @param created the {@link Calendar} representing when the payload was created
     * @return this PayloadInfo object
     */
    PayloadInfo setCreated(Calendar created);

    /**
     * Returns the ID of the user who created the payload
     * @return the ID of the user who create the payload
     */
    String getCreatedBy();

    /**
     * Set the ID of the user who created the payload.
     * @param createdBy the ID of the user who created the payload
     * @return this PayloadInfo object
     */
    PayloadInfo setCreatedBy(String createdBy);

    /**
     * Return the date and time when the payload was last modified
     * @return the {@link Calendar} representing when the payload was last modified
     */
    Calendar getLastModified();

    /**
     * Set the date and time when the payload was last modified
     * @param lastModified the {@link Calendar} representing when the payload was last modified
     * @return this PayloadInfo object
     */
    PayloadInfo setLastModified(Calendar lastModified);

    /**
     * Returns the ID of the user who last modified the payload.
     * @return the ID of the user who last modified the payload.
     */
    String getLastModifiedBy();

    /**
     * Set the ID of the user who last modified the payload.
     * @param lastModifiedBy the ID of the user who last modified the payload
     * @return this PayloadInfo objet
     */
    PayloadInfo setLastModifiedBy(String lastModifiedBy);

    /**
     * Return whether or not the payload is locked.
     * @return whether or not the payload is locked
     */
    boolean isLocked();

    /**
     * Set whether or not the payload is locked.
     * @param isLocked whether or not the payload is locked
     * @return this PayloadInfo object
     */
    PayloadInfo setIsLocked(boolean isLocked);

    /**
     * Returns the ID of the user who locked the payload.
     * @return the ID of the user who locked the payload
     */
    String getLockedBy();

    /**
     * Set the ID of the user who locked the payload.
     * @param lockedBy the ID of the user who locked the payload
     * @return this PayloadInfo object
     */
    PayloadInfo setLockedBy(String lockedBy);

    /**
     * Return the date and time that the payload will be turned on.  Generally only used by payloads of type
     * {@link PAYLOAD_TYPE_PAGE}.
     * @return the {@link Calendar} representing the date and time that the payload will be turned on
     */
    Calendar getOnTime();

    /**
     * Set the date and time that the payload will be turned on.  Generally only used by payloads of type
     * {@link PAYLOAD_TYPE_PAGE}.
     * @param onTime the {@link Calendar} representing the date and time that the payload will be turned on
     * @return this PayloadInfo object
     */
    PayloadInfo setOnTime(Calendar onTime);

    /**
     * Return the date and time that the payload will be turned off.  Generally only used by payloads of type
     * {@link PAYLOAD_TYPE_PAGE}.
     * @return the {@link Calendar} representing the date and time that the payload will be turned off
     */
    Calendar getOffTime();

    /**
     * Set the date and time that the payload will be turned off.  Generally only used by payloads of type
     * {@link PAYLOAD_TYPE_PAGE}.
     * @param offTime the {@link Calendar} representing the date and time that the payload will be turned off
     * @return this PayloadInfo object
     */
    PayloadInfo setOffTime(Calendar offTime);

    /**
     * Return the amount of time in ms until the payload is considered valid.  Generally only used by payloads
     * of type {@link PAYLOAD_TYPE_PAGE}.
     * @return the amount of time in ms until the payload is considered valid
     */
    long getTimeUntilValid();

    /**
     * Set the amount of time in ms until the payload is considered valid.  Generally only used by payloads of type
     * {@link PAYLOAD_TYPE_PAGE}
     * @param timeUntilValid the amount of time in ms until the payload is considered valid
     * @return this PayloadInfo object
     */
    PayloadInfo setTimeUntilValid(long timeUntilValid);

    /**
     * Return the number of monthly hits the payload has received.  Generally only used by payloads of type
     * {@link PAYLOAD_TYPE_PAGE}.
     * @return the number of monthly hits the payload has received
     */
    long getMonthlyHits();

    /**
     * Set the number of monthly hits the payload has received.  Generally only used by payloads of type
     * {@link PAYLOAD_TYPE_PAGE}.
     * @param monthlyHits the number of monthly hits the payload has received
     * @return this PayloadInfo object
     */
    PayloadInfo setMonthlyHits(long monthlyHits);

    /**
     * Return the description of the payload.
     * @return the description of the payload
     */
    String getDescription();

    /**
     * Set the description of the payload.
     * @param description the description of the payload
     * @return this PayloadInfo object
     */
    PayloadInfo setDescription(String description);

    /**
     * Return the path to the asset folder associated with the payload.  Generally only used by payloads of type
     * {@link PAYLOAD_TYPE_PROJECT}.
     * @return the path to the asset folder associated with the payload
     */
    String getAssetFolderPath();

    /**
     * Set the path to the asset folder associated with the payload.  Generally only used by payloads of type
     * {@link PAYLOAD_TYPE_PAGE}.
     * @param assetFolderPath the path to the asset folder associated with the payload
     * @return this PayloadInfo object
     */
    PayloadInfo setAssetFolderPath(String assetFolderPath);

    /**
     * Return whether or not the payload is active.  Generally only used by payloads of type
     * {@link PAYLOAD_TYPE_PAGE}.
     * @return whether or not the payload is active
     */
    boolean isActive();

    /**
     * Set whether or not the payload is active.  Generally only used by payloads of type {@link PAYLOAD_TYPE_PAGE}.
     * @param isActive whether or not the paylaod is active
     * @return this PayloadInfo object
     */
    PayloadInfo setIsActive(boolean isActive);

    /**
     * Return the custom property Map.  Custom properties are used to store information about a payload beyond the
     * most common properties.
     * @return the custom property {@link Map<String, Object>}
     */
    Map<String, Object> getCustomProperties();

    /**
     * Set the custom property Map.  Custom properties are used to store information about a payload beyond the most
     * common properties.
     * @param customProperties the custom property {@link Map<String, Object>}
     * @return this PayloadInfo object
     */
    PayloadInfo setCustomProperties(Map<String, Object> customProperties);

    /**
     * Add a property to the custom property {@link Map<String, Object>}
     * @param key the property's key
     * @param value the property's value
     * @return this PayloadInfo object
     */
    PayloadInfo addCustomProperty(String key, Object value);

    /**
     * Return a custom property from the custom property {@link Map<String, Object>}.
     * @param key the key to look up in the custom property Map
     * @return the value for the property or null if the key isn't found in the Map
     */
    Object getCustomProperty(String key);
}
