/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.reef.siren.builder;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import com.adobe.reef.siren.Action;
import com.adobe.reef.siren.EmbeddedLink;
import com.adobe.reef.siren.EmbeddedRepresentation;
import com.adobe.reef.siren.Entity;
import com.adobe.reef.siren.Link;

/**
 * {@code EntityBuilder} is a {@link Builder} implementation for {@Entity}.
 */
public class EntityBuilder extends Builder<Entity> {

    private String[] clazz;
    
    private String title;

    private String[] rel;

    private String href;

    private Map<String, Object> properties = new HashMap<String,Object>();

    private List<Entity> entities = new LinkedList<Entity>();

    private List<Link> links = new LinkedList<Link>();

    private List<Action> actions = new LinkedList<Action>();

    public EntityBuilder() {

    }

    public EntityBuilder setTitle(String title) {
        this.title = title;
        return this;
    }

    public EntityBuilder setClass(String[] clazz) {
        this.clazz = clazz;
        return this;
    }

    public EntityBuilder setRel(String[] rel) {
        this.rel = rel;
        return this;
    }

    public EntityBuilder setHref(String href) {
        this.href = href;
        return this;
    }

    public EntityBuilder setProperties(Map<String, Object> properties) {
        this.properties = properties;
        return this;
    }

    public EntityBuilder addProperty(String name, Object value) {
        this.properties.put(name, value);
        return this;
    }

    public EntityBuilder setEntities(List<Entity> entities) {
        this.entities = entities;
        return this;
    }

    public EntityBuilder addEntity(Entity entity) {
        this.entities.add(entity);
        return this;
    }

    public EntityBuilder setLinks(List<Link> links) {
        this.links = links;
        return this;
    }

    public EntityBuilder addLink(Link link) {
        this.links.add(link);
        return this;
    }

    public EntityBuilder setActions(List<Action> actions) {
        this.actions = actions;
        return this;
    }

    public EntityBuilder addAction(Action action) {
        this.actions.add(action);
        return this;
    }

    public EntityBuilder clear() {
        clazz = null;
        title = null;
        rel = null;
        href = null;
        properties.clear();
        entities.clear();
        links.clear();
        actions.clear();
        return this;
    }

    protected Entity doBuild() throws BuilderException {
        try {
            Entity entity = null;
            if (href != null) {
                entity = new EmbeddedLink(rel, href);
            } else 
            if (rel != null) {
                entity = new EmbeddedRepresentation(rel);
            } else {
                entity = new Entity();
            }
            entity.setClazz(clazz);
            entity.setTitle(title);
            entity.setRel(rel);
            entity.setHref(href);
            entity.setProperties(properties);
            entity.setEntities(entities);
            entity.setLinks(links);
            entity.setActions(actions);
            return entity;
        } catch (IllegalArgumentException e) {
            throw new BuilderException(e.getMessage(), e);
        }
    }

    protected void validate(Entity entity) throws BuilderValidationException {
        if (entity instanceof EmbeddedLink) {
            validateHref(entity);
            validateRel(entity);
        } else if (entity instanceof EmbeddedRepresentation) {
            validateRel(entity);
            validateSelf(entity);
        } else {
            validateSelf(entity);
        }
    }

    private void validateHref(Entity entity) throws BuilderValidationException {
        if (entity.getHref() == null) {
            throw new BuilderValidationException("Attribute 'href' cannot be null or empty.");
        } else {
            try {
                new URI(entity.getHref());
            } catch (URISyntaxException e) {
                throw new BuilderValidationException("Entity href is not a valid URI.");
            }
        }
    }

    private void validateRel(Entity entity) throws BuilderValidationException {
        if (entity.getRel() == null) {
            throw new BuilderValidationException("Attribute 'rel' cannot be null or empty.");
        }
    }

    private void validateSelf(Entity entity) throws BuilderValidationException {
        if (!entity.getLinks().isEmpty()) {
            for (Link link : entity.getLinks()) {
                List<String> rels = Arrays.asList(link.getRel());
                if (rels.contains("self")) {
                    try {
                        new URI(link.getHref());
                        return;
                    } catch (URISyntaxException e) {
                        throw new BuilderValidationException("Link href is not a valid URI.");
                    }
                }
            }
        }
        throw new BuilderValidationException("Entity has no valid 'self' link.");
    }

}
