// =================================================================================================
// ADOBE SYSTEMS INCORPORATED
// Copyright 2012 Adobe Systems Incorporated
// All Rights Reserved
//
// NOTICE:  Adobe permits you to use, modify, and distribute this file in accordance with the terms
// of the Adobe license agreement accompanying it.
// =================================================================================================
package com.adobe.xmp.core;

/**
 * This represents a array in the XMP tree. There are 3 types of XMP arrays: unordered, ordered or alternative
 *
 */
public interface XMPArray extends XMPNode
{
	/**
	 * The different types that an XMP Array can be
	 *
	 */
	public enum Form
	{
		UNORDERED, ORDERED, ALTERNATIVE
	}
	
	/**
	 * Returns the array Form (Unordered, ordered or alternative)
	 * @return the array Form
	 */
	public Form getForm();
		
	/**
	 * Delete all array elements
	 */
	void clear();
	
	/**
	 * Checks if the array is empty
	 * @return True if the array is empty False otherwise
	 */
	boolean isEmpty();
	
	/**
	 * Returns the base item at position index (without knowing the special type)
	 * @param index the index of the array item to get
	 * @return the array item at position index (or throws IndexOutOfBoundsException if 
	 *         there is no node at position index)
	 * @throws IndexOutOfBoundsException
	 */
	XMPNode get( int index );

	/**
	 * Remove a node from position index
	 * @param index the index of the element to remove
	 * @return the removed child element
	 * @throws IndexOutOfBoundsException if the index is out of range (index < 0 || index >= size())
	 */
	XMPNode remove( int index );
	
	// special get
	/**
	 * Get simple array item at position index
	 * @param index the index for the item to get
	 * @return the simple property or null if there was nothing to return or the type was wrong
	 */
	XMPSimple getSimple( int index );
	
	/**
	 * Get a struct array item at position index
	 * @param index the index for the item to get
	 * @return the struct property or null if there was nothing to return or the type was wrong
	 */
	XMPStruct getStruct( int index );
	
	/**
	 * Get nested array property in this array at position index
	 * @param index the index for the item to get
	 * @return the array or null if there was nothing to return or the type was wrong
	 */
	XMPArray getArray( int index );
	
	/**
	 * Get language alternative property in this array at position index
	 * @param index the index for the item to get
	 * @return the array or null if there was nothing to return or the type was wrong
	 */
	XMPLanguageAlternative getLanguageAlternative( int index );
	
	// Append to end of list
	/**
	 * Append new simple property to the end of the list
	 * @param value the value of the new property (namespace is the same as the host array)
	 * @return the newly appended property
	 */
	XMPSimple appendSimple( String value );
	
	/**
	 * Append new empty struct property to the end of the list
	 * @return the newly appended property
	 */
	XMPStruct appendStruct();
	
	/**
	 * Append new empty array to the end of the list
	 * @param form the array form
	 * @return the newly appended property
	 */
	XMPArray appendArray( Form form );

	/**
	 * Append new language alternative to the end of the list
	 * @return the newly appended property
	 */
	XMPLanguageAlternative appendLanguageAlternative( );
			
	// Replace existing or create
	/**
	 * Sets a new simple property at position index. This will overwrite any existing property at this index. 
	 * If the index is equal to the size of the list, it will be appened.
	 * @param index the index to set the property
	 * @param value the value of the property to set
	 * @return the new property at position index or null if nothing was overwritten/appended
	 */
	XMPSimple setSimple( int index, String value );
	
	/**
	 * Sets a new struct  at position index. This will overwrite any existing property at this index. 
	 * If the index is equal to the size of the list, it will be appened.
	 * @param index the index to set the property
	 * @return the new property at position index or null if nothing was overwritten/appended
	 */
	XMPStruct setStruct( int index );
	
	/**
	 * Sets a new array at position index. This will overwrite any existing property at this index. 
	 * If the index is equal to the size of the list, it will be appened.
	 * @param index the index to set the array
	 * @param form the array form
	 * @return the new array at position index or null if nothing was overwritten/appended
	 */
	XMPArray setArray( int index, Form form );
	
	/**
	 * Sets a new language alternative at position index. This will overwrite any existing property at this index. 
	 * If the index is equal to the size of the list, it will be appened.
	 * @param index the index to set the language alternative
	 * @return the new language alternative at position index or null if nothing was overwritten/appended
	 */
	XMPLanguageAlternative setLanguageAlternative( int index );
	
	// Insert new and shift rest
	/**
	 * Insert new simple property at position index and shift other array items to the back. The insert at the end
	 * of the list is similar to an append.
	 * @param index the index for the insert
	 * @param value the value for the inserted new simple property
	 * @return the newly inserted simple property or null if nothing was inserted
	 */
	XMPSimple insertSimple( int index, String value );
	
	/**
	 * Insert new struct property at position index and shift other array items to the back. The insert at the end
	 * of the list is similar to an append.
	 * @param index the index for the insert
	 * @return the newly inserted struct property or null if nothing was inserted
	 */
	XMPStruct insertStruct( int index );
	
	/**
	 * Insert new array at position index and shift other array items to the back. The insert at the end
	 * of the list is similar to an append.
	 * @param index the index for the insert
	 * @param form the array form
	 * @return the newly inserted array or null if nothing was inserted
	 */
	XMPArray insertArray( int index, Form form );
	
	/**
	 * Insert new language alternative at position index and shift other array items to the back. The insert at the end
	 * of the list is similar to an append.
	 * @param index the index for the insert
	 * @return the newly inserted array or null if nothing was inserted
	 */
	XMPLanguageAlternative insertLanguageAlternative( int index );
}
