// =================================================================================================
// ADOBE SYSTEMS INCORPORATED
// Copyright 2012 Adobe Systems Incorporated
// All Rights Reserved
//
// NOTICE:  Adobe permits you to use, modify, and distribute this file in accordance with the terms
// of the Adobe license agreement accompanying it.
// =================================================================================================
package com.adobe.xmp.core;

import com.adobe.xmp.path.XMPPath;

/**
 * This interface defines how {@link XMPPath} is used to address nodes in this {@link XMPNode}. 
 */
interface XMPPathAddressable
{
	// Path APIs
	/**
	 * Return the path of this XMPNode
	 * @return the path of this XMPNode
	 */
	XMPPath getXMPPath();
	
	
	/**
	 * Return the node of the given path, relative to this node.
	 * @param path the path to the desired node relative to this node
	 * @return the node at the path position or null if it could not be found
	 */
	XMPNode get( XMPPath path );
	
	/**
	 *  Remove a node at a given path, relative to this node
	 * @param path the path to the node that shall be deleted
	 * @return the removed node or null, if nothing is removed
	 * @throws XMPException throws in case the remove path points to 
	 * an xml:lang qualifier of a Language Alternative
	 */

	XMPNode remove( XMPPath path ) throws XMPException;
		
	/**
	 * Returns a simple property from the given path, relative to this node.
	 * @param path the path to the simple property, relative to this node.
	 * @return the desired simple property or null if there was none (or the type is different)
	 */
	XMPSimple getSimple( XMPPath path );
		
	/**
	 * Returns a struct property from the given path, relative to this node.
	 * @param path the path to the struct property, relative to this node.
	 * @return the desired struct property or null if there was none (or the type is different)
	 */
	XMPStruct getStruct( XMPPath path );
	
	/**
	 * Returns an array from the given path, relative to this node.
	 * @param path the path to the array, relative to this node.
	 * @return the desired array or null if there was none (or the type is different)
	 */
	XMPArray getArray( XMPPath path );
	
	/**
	 * Returns a language Alternative from the given path, relative to this node.
	 * @param path the path to the langAlt, relative to this node.
	 * @return the desired langAlt or null if there was none (or the type is different)
	 */
	XMPLanguageAlternative getLanguageAlternative( XMPPath path );
	
	/**
	 * returns a human readable version of this node tree (recursive)
	 * @return returns a human readable version of this node tree (recursive)
	 */
	String dump();
	
}
