// =================================================================================================
// ADOBE SYSTEMS INCORPORATED
// Copyright 2012 Adobe Systems Incorporated
// All Rights Reserved
//
// NOTICE:  Adobe permits you to use, modify, and distribute this file in accordance with the terms
// of the Adobe license agreement accompanying it.
// =================================================================================================
package com.adobe.xmp.core.serializer;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Map;

import com.adobe.xmp.core.XMPException;
import com.adobe.xmp.core.XMPMetadata;

/**
 * Interface for all kinds of serializers for the XMP data model.
 */
public interface XMPSerializer 
{
	/**
	 * Serialize XMP data as RDF/XML to an OutputStream
	 * 
	 * @param xmp the XMP object to serialize
	 * @param out the stream to write to. Client has to open/close the stream.
	 * @param serializeContext an optional serialization context that 
	 * 		  can contains additional information to configure the serialization.
	 * @throws IOException If an error occurred writing to the stream
	 * @throws XMPException This exception is thrown if the data model contains portions that can't be serialized 
	 * 		   by a specific serializer or if the options are inconsistent.
	 */
	void serialize(XMPMetadata xmp, OutputStream out, Map<String, Object> serializeContext) throws IOException, XMPException;
	
	/**
	 * Serialize XMP data as RDF/XML into a String
	 * 
	 * @param xmp the XMP object to serialize
	 * @param serializeContext an optional serialization context that 
	 * 		  can contains additional information to configure the serialization.
	 * @return String containing the XMP data as RDF/XML
 	 * @throws IOException If an error occurred writing to the internal stream or out of memory
	 * @throws XMPException This exception is thrown if the data model contains portions that can't be serialized 
	 * 		   by a specific serializer or if the options are inconsistent.
	 */
	String serializeToString(XMPMetadata xmp, Map<String, Object> serializeContext) throws IOException, XMPException;
	
	/**
	 * Serialize XMP data as RDF/XML UTF-8 encoded into a byte buffer
	 * @param xmp the XMP object to serialize
	 * @param serializeContext an optional serialization context that 
	 * 		  can contains additional information to configure the serialization.
	 * @return Buffer containing the XMP data as RDF/XML
 	 * @throws IOException If an error occurred writing to the stream
	 * @throws XMPException This exception is thrown if the data model contains portions that can't be serialized 
	 * 		   by a specific serializer or if the options are inconsistent.
	 */
	byte[] serializeToBuffer(XMPMetadata xmp, Map<String, Object> serializeContext) throws IOException, XMPException;
}
