// =================================================================================================
// ADOBE SYSTEMS INCORPORATED
// Copyright 2012 Adobe Systems Incorporated
// All Rights Reserved
//
// NOTICE:  Adobe permits you to use, modify, and distribute this file in accordance with the terms
// of the Adobe license agreement accompanying it.
// =================================================================================================
package com.adobe.xmp.path;

/**
 * Represents one segment in a path into the XMP tree.
 * Each segment has to be created through one of the factories and is immutable.
 */
public class XMPPathSegment 
{
	/**
	 * Types of path segments
	 */
	public enum Type
	{
		PROPERTY,			// Any property that consists of namespace and a localname 
		ARRAY_INDEX, 		// An array index which does not have a namespace or localname itself
		QUALIFIER,			// qualifier of a property, also consists of namespace and localname
		QUALIFIER_SELECTOR 	// selects a specific qualifier by its value (e.g. specific language)
	}
	
	private final Type type;
	private final String namespace;
	private final String name;
	private final String value;
	private final int index;

	// Private constructors
	private XMPPathSegment(Type type, String namespace, String name )
	{
		this(type, namespace, name, null, -1 );
	}
	
	private XMPPathSegment(Type type, String namespace, String name, String value )
	{
		this(type, namespace, name, value, -1 );
	}
	
	private XMPPathSegment(Type type, String namespace, String name, int index )
	{
		this(type, namespace, name, null, index );
	}
	
	private XMPPathSegment(Type type, String namespace, String name, String value, int index)
	{
		this.type = type;
		this.namespace = namespace;
		this.name = name;
		this.value = value;
		this.index = index;
	}

	// Factories to create the specific segments
	/**
	 * Creates a normal property path segment. 
	 * These are essentially all properties (simple, struct and arrays).
	 * @param namespace Namespace URI of the property
	 * @param name Local name of the property
	 * @return a new immutable path segment
	 */
	public static XMPPathSegment createPropertySegment( String namespace, String name )
	{
		return new XMPPathSegment( Type.PROPERTY, namespace, name );
	}
	
	/**
	 * Creates an array index path segment that denotes a specific element of an array.
	 * Such segments do not have an own name and inherits the namespace from the Array property itself
	 * @param namespace Namespace URI of the array property
	 * @param index Index of the element
	 * @return a new immutable path segment
	 */
	public static XMPPathSegment createArrayIndexSegment( String namespace, int index )
	{
		return new XMPPathSegment( Type.ARRAY_INDEX, namespace, null, index );
	}
	
	/**
	 * Creates a Qualifier path segment, which behaves like a normal property
	 * @param namespace Namespace URI of the qualifier property
	 * @param name Local name of the qualifier property
	 * @return a new immutable path segment
	 */
	public static XMPPathSegment createQualifierSegment(String namespace, String name)
	{
		return new XMPPathSegment( Type.QUALIFIER, namespace, name );
	}
	
	/**
	 * Creates a path segment that selects a specific qualifier by its value.
	 * For example a specific language in a alternative array of languages.
	 * @param namespace Namespace URI of the property
	 * @param name Local name of the property
	 * @param value The value that depicts the specific qualifier
	 * @return a new immutable path segment
	 */
	public static XMPPathSegment createQualifierSelectorSegment(String namespace, String name, String value)
	{
		return new XMPPathSegment( Type.QUALIFIER_SELECTOR, namespace, name, value );
	}

	// Getter
	/**
	 * @return the namespace
	 */
	public String getNamespace()
	{
		return namespace;
	}

	/**
	 * @return the name
	 */
	public String getName()
	{
		return name;
	}

	/**
	 * @return the type
	 */
	public Type getType()
	{
		return type;
	}
	
	public int getIndex()
	{
		return index;
	}

	/**
	 * @return the value
	 */
	public String getValue()
	{
		return value;
	}

	/**
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + index;
		result = prime * result + ((name == null) ? 0 : name.hashCode());
		result = prime * result
				+ ((namespace == null) ? 0 : namespace.hashCode());
		result = prime * result + ((type == null) ? 0 : type.hashCode());
		result = prime * result + ((value == null) ? 0 : value.hashCode());
		return result;
	}

	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) 
	{
		if (this == obj)
		{
			return true;
		}
		if (obj == null)
		{
			return false;
		}
		if (getClass() != obj.getClass())
		{
			return false;
		}
		
		XMPPathSegment other = (XMPPathSegment) obj;
		
		if (type != other.type)
		{
			return false;
		}
		
		switch(type)
		{
		case QUALIFIER_SELECTOR:
			if (value == null) 
			{
				if (other.value != null)
				{
					return false;
				}
			} 
			else if (!value.equals(other.value))
			{
				return false;
			}
			// break is intentionally missing
		case PROPERTY: 
		case QUALIFIER:
			if (name == null) 
			{
				if (other.name != null)
				{
					return false;
				}
			} 
			else if (!name.equals(other.name))
			{
				return false;
			}
			
			if (namespace == null) 
			{
				if (other.namespace != null)
				{
					return false;
				}
			} 
			else if (!namespace.equals(other.namespace))
			{
				return false;
			}
			
			break;
		case ARRAY_INDEX:
			if (namespace == null) 
			{
				if (other.namespace != null)
				{
					return false;
				}
			} 
			else if (!namespace.equals(other.namespace))
			{
				return false;
			}
			
			if (index != other.index)
			{
				return false;
			}
			
			break;

		}
		
		return true;
	}
	
}
