// =================================================================================================
// ADOBE SYSTEMS INCORPORATED
// Copyright 2011 Adobe Systems Incorporated
// All Rights Reserved
//
// NOTICE:  Adobe permits you to use, modify, and distribute this file in accordance with the terms
// of the Adobe license agreement accompanying it.
// =================================================================================================

package com.adobe.xmp.schema.model;

import java.util.List;

/**
 * Describes an XMP array property.
 * There are four types of arrays:
 * <ul>
 * 		<li>Bag</li> (unorderen array)
 * 		<li>Seq</li> (ordered array)
 * 		<li>Alt</li> (alternative array)
 * 		<li>Alt-Text</li> (language alternative, syntactically equal to Alt,
 * 			but contains only text properties with a language qualifier)
 * </ul>
 * Additionally, a rule can be added to contrain an array to have
 * a minimum and/or maximum number of Items (LengthRule).
 *
 * @author Stefan Makswit
 */
public interface StructType extends PropertyType
{
	/**
	 * The default namespace is the namespace that most struct fields are in.
	 * All structs defined in the XMP Specifications contain field only from a single namespace.
	 * Though, the XMP data model allows structs that contain fields of multiple namespaces.
	 * @return Returns the default namespace of the struct.
	 */
	String getDefaultNamespaceURI();
	
	/**
	 * Adds a struct field. If the field has no namespace defined,
	 * it inherits the default namespace of the struct.
	 * 
	 * @param field the struct field to add
	 */
	void addField(PropertyDescription field);
	
	/**
	 * Returns a struct field by namespace and local name.
	 * 
	 * @param namespaceURI a namespace
	 * @param localName the name of a struct field
	 * @return Returns the struct field if found, null otherwise.
	 */
	PropertyDescription getField(String namespaceURI, String localName);
		
	/**
	 * @return Returns a collection of all struct fields.
	 * 		The order of the added fields should be kept.
	 */
	List<PropertyDescription> getFields();
	
	/**
	 * Removes a struct field by namespace and local name.
	 * If it is not existing, the call is ignored.
	 * 
	 * @param namespaceURI a namespace
	 * @param localName the name of the struct field to delete
	 */
	void removeField(String namespaceURI, String localName);
}
