// =================================================================================================
// ADOBE SYSTEMS INCORPORATED
// Copyright 2011 Adobe Systems Incorporated
// All Rights Reserved
//
// NOTICE:  Adobe permits you to use, modify, and distribute this file in accordance with the terms
// of the Adobe license agreement accompanying it.
// =================================================================================================

package com.adobe.xmp.schema.model;

import java.util.Collection;
import java.util.List;

import com.adobe.xmp.schema.model.ArrayType.ArrayForm;
import com.adobe.xmp.schema.model.SimpleType.BasicType;


/**
 * The type registry serves two purposes:
 * 
 * 1. It contains factory methods to create the different property types.
 * 
 * 2.It optionally registers XMP types with names and makes them re-usable this
 * way. The "named" types are also needed to reference predefined types in our
 * fragment libraries. The registering part is not necessarily needed to create
 * a schema.
 * 
 * @author Stefan Makswit
 */
public interface TypeRegistry
{
	/**
	 * Returns a registered type. This type must most be modified anymore,
	 * once it has been registered.
	 * 
	 * @param typeName
	 *            name of the Type
	 * @return Returns the type if available in the registry otherwise
	 *         <code>null</code>
	 */
	PropertyType getType(String typeName);


	/**
	 * @param type Enum value of the basic type
	 * @return Returns the default implementation of the the basic type
	 */
	PropertyType getType(BasicType type);


	/**
	 * @return Returns a collection of all registered Types
	 */
	List<PropertyType> getTypes();


	/**
	 * Registers a property type by name.
	 * Existing types must not be overwritten;
	 * array types cannot be registered
	 * 
	 * @param name the name of the property
	 * @param propertyType
	 *            the PropertyType to register
	 * @throws XMPSchemaException Thrown if the name is not valid 
	 * 		or it is tried to register duplicate names.
	 */
	void registerType(String name, PropertyType propertyType) throws XMPSchemaException;

	

	/**
	 * Creates a derived simple type from a basic type.
	 * 
	 * @param basicType one of the five basic types: Text, Boolean, Integer, Real or Date 
	 * @return Returns the the derived type.
	 */
	SimpleType createDerivedType(BasicType basicType);
	

	/**
	 * Creates a derived simple type.
	 * These are all types that add rules or other semantics to the basic types.
	 * 
	 * @param inheritedType
	 *            any simple type
	 * @return Returns a new simple type.
	 */
	SimpleType createDerivedType(SimpleType inheritedType);

	
	/**
	 * Creates a struct type.
	 * 
	 * @param namespaceURI a namespace
	 * @return Returns a new struct type
	 */
	StructType createStruct(String namespaceURI);


	/**
	 * Creates an array type. Array types cannot have a name, because they are
	 * rather defined by their item type.
	 * 
	 * @param form
	 *            the form of the array
	 * @param itemType
	 *            the type of the array items
	 * @return Returns a new array type
	 */
	ArrayType createArray(ArrayForm form, PropertyType itemType);


	/**
	 * Registers an unspecified type, that is just a place holder for a type that is defined later. 
	 * This should be used only by schema parsers that want to do late binding on some types.
	 * 
	 * @param typeName the name of the type 
	 * @return Returns an "unspecified" type.
	 */
	PropertyType registerUnspecifiedType(String typeName);


	/**
	 * @return Returns the list of unspecified types. Should be empty when the parsing is done.
	 */
	Collection<UnspecifiedType> getUnspecifiedTypes();


	/**
	 * Replaces the type of all properties that reference the unspecified type.
	 * Afterwards the unspecified type is deleted.
	 * 
	 * @param unspecified
	 *            an unspecified property type
	 * @param finalType
	 *            the final property type.
	 */
	void replaceUnspecifiedType(PropertyType unspecified,
			PropertyType finalType);
}