/*
 * File: PropertyInfo.java
 * ************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.xmp.schema.rng.model;

import java.util.HashMap;
import java.util.Map;
import java.util.Stack;

import javax.xml.namespace.QName;

import com.adobe.xmp.schema.model.ArrayType.ArrayForm;
import com.adobe.xmp.schema.rng.model.ParamInfoGroup.Operator;
import com.adobe.xmp.schema.rng.parser.annotation.RNGDecorationAnnotation.ACCESS;
import com.adobe.xmp.schema.rng.parser.annotation.RNGDecorationAnnotation.CHOICE;

/**
 * This class holds relevant data for constructing a {@link com.adobe.xmp.schema.model.PropertyDescription} object collected from parsing the
 * RelaxNG schema.
 * 
 * @author pwollek
 */
public class PropertyInfo implements PropertyAnnotationInfo
{
	private String mNS;

	private String mName;

	private ParamInfoGroup mParamInfos;

	private DatatypeInfo mRawDataType;

	private DatatypeInfo mDataType;

	private Stack<PropertyInfo> mFields; // struct fields

	private Stack<PropertyInfo> mQualifiers; // qualifiers

	private ArrayForm mArrayForm;

	private String mLabel;

	private String mDescription;

	private ACCESS mAccess;

	private CHOICE choice;

	private boolean mDeprecated;

	private boolean mMandatory;

	private boolean mIsQualifier;
	
	private boolean mIsArrayItemQualifier;

	private SchemaInfo mSchemaInfo;

	/**
	 * Store custom decorators with arbitrary QNames, that contain a property-value list.
	 * The outer Map maps the QName to a second property-value Map, that contains the actual decorator properties.
	 */
	private HashMap<QName, Map<String, String>> mDecoratorStore;
	
	/**
	 * 
	 * Constructs a new PropertyInfo.
	 * 
	 * @param ns
	 *            Namespace of the property
	 * @param name
	 *            Name of the property
	 * @param schemaInfo
	 *            {@link SchemaInfo} object, which stores information retrieved from annotations about the XMP schema
	 */
	public PropertyInfo(String ns, String name, SchemaInfo schemaInfo)
	{
		mNS = ns;
		mName = name;
		mParamInfos = new ParamInfoGroup(Operator.kAND);
		// Raw data type by default is set to text
		mRawDataType = new DatatypeInfo("Text", true);
		mFields = new Stack<PropertyInfo>();
		mQualifiers = new Stack<PropertyInfo>();
		mArrayForm = null;
		mLabel = "";
		mDescription = "";
		mAccess = null;
		mDeprecated = false;
		mMandatory = true;
		mIsQualifier = false;
		mIsArrayItemQualifier = false;
		mSchemaInfo = schemaInfo;
	}

	/**
	 * Adds a child, which could either be a property qualifier, array item qualifier or a struct field
	 * 
	 * @param child
	 *            child to add
	 */
	public void addChild(PropertyInfo child)
	{
		if (child.isQualifier() || child.isArrayItemQualifier())
		{
			mQualifiers.add(child);
		}
		else
		{
			mFields.add(child);
		}
	}

	/**
	 * @return List of qualifiers of this {@link PropertyInfo}
	 */
	public Stack<PropertyInfo> getQualifiers()
	{
		return mQualifiers;
	}

	/**
	 * @return List of fields of this {@link PropertyInfo}. Relevant only for struct property.
	 */
	public Stack<PropertyInfo> getFields()
	{
		return mFields;
	}

	/**
	 * @param mNS
	 *            the mNS to set
	 */
	public void setNS(String mNS)
	{
		this.mNS = mNS;
	}

	/**
	 * @return the mNS
	 */
	public String getNS()
	{
		return mNS;
	}

	/**
	 * @param mName
	 *            the mName to set
	 */
	public void setName(String mName)
	{
		this.mName = mName;
	}

	/**
	 * @return the mName
	 */
	public String getName()
	{
		return mName;
	}

	/**
	 * @param mParamInfos
	 *            the mParamInfos to set
	 */
	public void setParamInfos(ParamInfoGroup mParamInfos)
	{
		if(this.mParamInfos == null || this.mParamInfos.size() == 0)
		{
			this.mParamInfos = mParamInfos;
		}
		else
		{
			this.mParamInfos.push(mParamInfos);
		}
	}

	/**
	 * @return the mParamInfos
	 */
	public ParamInfoGroup getParamInfos()
	{
		return mParamInfos;
	}

	/**
	 * @param mRawDataType
	 *            the mRawDataType to set
	 */
	public void setRawDataType(DatatypeInfo mRawDataType)
	{
		this.mRawDataType = mRawDataType;
	}

	/**
	 * @return the mRawDataType
	 */
	public DatatypeInfo getRawDataType()
	{
		return mRawDataType;
	}

	/**
	 * @param mDataType
	 *            the mDataType to set
	 */
	public void setDataType(DatatypeInfo mDataType)
	{
		this.mDataType = mDataType;
	}

	/**
	 * @return the mDataType
	 */
	public DatatypeInfo getDataType()
	{
		return mDataType;
	}

	/**
	 * @param form
	 *            the mArrayForm to set
	 */
	public void setArrayForm(ArrayForm form)
	{
		this.mArrayForm = form;
	}

	/**
	 * @return the mArrayForm
	 */
	public ArrayForm getArrayForm()
	{
		return mArrayForm;
	}

	/**
	 * @param mLabel
	 *            the mLabel to set
	 */
	public void setLabel(String mLabel)
	{
		this.mLabel = mLabel;
	}

	/**
	 * @return the mLabel
	 */
	public String getLabel()
	{
		return mLabel;
	}

	/**
	 * @param mDescription
	 *            the mDescription to set
	 */
	public void setDescription(String mDescription)
	{
		this.mDescription = mDescription;
	}

	/**
	 * @return the mDescription
	 */
	public String getDescription()
	{
		return mDescription;
	}

	/**
	 * @param mAccess
	 *            the mAccess to set
	 */
	public void setAccess(ACCESS mAccess)
	{
		this.mAccess = mAccess;
	}

	/**
	 * @return the mAccess
	 */
	public ACCESS getAccess()
	{
		return mAccess;
	}

	/**
	 * @param mDeprecated
	 *            the mDeprecated to set
	 */
	public void setDeprecated(boolean mDeprecated)
	{
		this.mDeprecated = mDeprecated;
	}

	/**
	 * @return the mDeprecated
	 */
	public boolean isDeprecated()
	{
		return mDeprecated;
	}

	/**
	 * @param mMandatory
	 *            the mMandatory to set
	 */
	public void setMandatory(boolean mMandatory)
	{
		this.mMandatory = mMandatory;
	}

	/**
	 * @return the mMandatory
	 */
	public boolean isMandatory()
	{
		return mMandatory;
	}
	
	/**
	 * Sets if this propertyInfo stores information for an array item qualifier.
	 * @param isArrayItemQualifier boolean
	 */
	public void setIsArrayItemQualifier(boolean isArrayItemQualifier)
	{
		this.mIsArrayItemQualifier = isArrayItemQualifier;
	}
	
	/**
	 * Gets if the propertyInfo holds information for an array item qualifier.
	 * @return boolean
	 */
	public boolean isArrayItemQualifier()
	{
		return mIsArrayItemQualifier;
	}
	

	/**
	 * @param isQualifier
	 *            the mIsQualifier to set
	 */
	public void setIsQualifier(boolean isQualifier)
	{
		this.mIsQualifier = isQualifier;
	}

	/**
	 * @return the mIsQualifier
	 */
	public boolean isQualifier()
	{
		return mIsQualifier;
	}

	/**
	 * @param mSchemaInfo
	 *            the mSchemaInfo to set
	 */
	public void setSchemaInfo(SchemaInfo mSchemaInfo)
	{
		this.mSchemaInfo = mSchemaInfo;
	}

	/**
	 * @return the mSchemaInfo
	 */
	public SchemaInfo getSchemaInfo()
	{
		return mSchemaInfo;
	}

	/**
	 * @param choice
	 *            the choice to set
	 */
	public void setChoice(CHOICE choice)
	{
		this.choice = choice;
	}

	/**
	 * @return the choice
	 */
	public CHOICE getChoice()
	{
		return choice;
	}
	
	/**
	 * Returns if this property has decorators
	 * @return boolean
	 */
	public boolean hasDecorators()
	{
		return mDecoratorStore != null && mDecoratorStore.size() > 0;
	}
	
	/**
	 * Gets the decorator store for this {@link PropertyInfo}
	 * @return HashMap
	 */
	public HashMap<QName, Map<String, String>> getDecoratorStore()
	{
		if (mDecoratorStore == null)
		{
			mDecoratorStore = new HashMap<QName, Map<String,String>>();
		}
		return mDecoratorStore;
	}
	
	/**
	 * Gets decorator map for given {@link QName}. If entry does not 
	 * exist for the given qname, an entry is created in the decoratorstore.
	 * @param qname qname
	 * @return Map
	 */
	public Map<String, String> getDecorator(QName qname)
	{
		Map<String, String> map = getDecoratorStore().get(qname);;
		if(map == null)
		{
			map = new HashMap<String, String>();
			mDecoratorStore.put(qname, map);
		}
		return map;
	}

	@Override
	public String toString()
	{
		StringBuilder str = new StringBuilder();
		str.append("\nName = ").append(mName);
		str.append(" - Ns = ").append(mNS);
		str.append(" - label = ").append(mLabel);
		str.append("\nDeprecated = ").append(mDeprecated);
		str.append("\nMandatory = ").append(mMandatory);
		if(mFields != null && mFields.size() > 0)
		{
			str.append("\n========= Fields  ===============");
			for(PropertyInfo p : mFields)
			{
				str.append(p.toString());
			}
			str.append("\n=================================");
		}
		if(mQualifiers != null && mQualifiers.size() > 0)
		{
			str.append("\n========= Qualifiers  ===============");
			for(PropertyInfo p : mQualifiers)
			{
				str.append(p.toString());
			}
			str.append("\n=================================");
		}
		
		if(mArrayForm != null)
		{
			str.append("\n Array Type " + mArrayForm);
		}
			
		if(mParamInfos != null)
		{
			str.append("\n========= Param Info  ===============");
			str.append("\n" + mParamInfos);
			str.append("\n=================================");
		}
		
		if(mDataType != null)
		{
			str.append("\n DataType" + mDataType);
		}
		
		if(mRawDataType != null)
		{
			str.append("\n Raw DataType" + mRawDataType);
		}
		
		return str.toString();
	}
	
}
