/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.analytics.sitecatalyst.util;

import java.util.Collections;

import org.apache.commons.httpclient.HostConfiguration;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpConnectionManager;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.SimpleHttpConnectionManager;
import org.apache.commons.httpclient.cookie.CookiePolicy;
import org.apache.commons.httpclient.params.HttpClientParams;
import org.apache.commons.httpclient.params.HttpConnectionManagerParams;
import org.apache.commons.httpclient.protocol.Protocol;
import org.apache.commons.httpclient.protocol.ProtocolSocketFactory;

/**
 * Utility methods for the Apache HTTPClient.
 */
public class HttpClientUtils {

    public static final int DEFAULT_CONNECTION_TIMEOUT = 30 * 1000;
    public static final int DEFAULT_MAX_CONNECTIONS = 50;
    public static final boolean DEFAULT_STALE_CONNECTION_CHECK = true;
    public static final int DEFAULT_SO_LINGER = 0;

    private HttpClientUtils() {
    }

    /**
     * Creates a new thread-safe HTTPClient.
     *
     * @return the new thread-safe HttpClient
     */
    public static HttpClient newMultiThreaded() {
        return newMultiThreaded(DEFAULT_CONNECTION_TIMEOUT, DEFAULT_MAX_CONNECTIONS);
    }

    /**
     * Creates a new thread-safe HTTPClient.
     *
     * @param connectionTimeout connection timeout in milliseconds
     * @param maxConnections max concurrent connections
     * @return the new thread-safe HttpClient
     */
    public static HttpClient newMultiThreaded(int connectionTimeout, int maxConnections) {
        return newMultiThreaded(connectionTimeout, maxConnections, DEFAULT_STALE_CONNECTION_CHECK, DEFAULT_SO_LINGER);
    }

    /**
     * Creates a new thread-safe HTTPClient.
     *
     * @param connectionTimeout connection timeout in milliseconds
     * @param maxConnections max concurrent connections
     * @param staleConnectionCheck if reused connections should be checked before using (recommended)
     * @param soLinger timeout before connection is closed forcefully (0 recommended,
     *                 otherwise the timeout is awaited when a broken connection is tried to be reused)
     * @return the new thread-safe HttpClient
     */
    public static HttpClient newMultiThreaded(int connectionTimeout, int maxConnections,
                                              boolean staleConnectionCheck, int soLinger) {
        HttpClientParams params = newParams(connectionTimeout, maxConnections, staleConnectionCheck, soLinger);
        HttpClient httpClient = new HttpClient(params, new MultiThreadedHttpConnectionManager());
        return httpClient;
    }

    private static HttpClientParams newParams(int connectionTimeout, int maxConnections,
                                              boolean staleConnectionCheck, int soLinger) {
        HttpClientParams params = new HttpClientParams();
        params.setCookiePolicy(CookiePolicy.IGNORE_COOKIES);

        params.setParameter(HttpConnectionManagerParams.CONNECTION_TIMEOUT, connectionTimeout);
        params.setParameter(HttpConnectionManagerParams.STALE_CONNECTION_CHECK, staleConnectionCheck);

        params.setParameter(HttpConnectionManagerParams.MAX_TOTAL_CONNECTIONS, maxConnections);
        params.setParameter(HttpConnectionManagerParams.MAX_HOST_CONNECTIONS,
                Collections.singletonMap(HostConfiguration.ANY_HOST_CONFIGURATION, maxConnections));

        params.setParameter(HttpConnectionManagerParams.SO_LINGER, soLinger);
        return params;
    }

    /**
     * Allows self signed certificates for HTTPS connections with the given host and port.
     *
     * @param httpClient httpClient
     * @param host host
     * @param port port
     */
    public static void allowSelfSigned(HttpClient httpClient, String host, int port) {
        if (port < 0) {
            port = 443;
        }
        Protocol httpsProtocol = new Protocol("https", (ProtocolSocketFactory) new EasySSLProtocolSocketFactory(true), port);
        httpClient.getHostConfiguration().setHost(host, port, httpsProtocol);
    }

    /**
     * Shuts down the HttpConnectionManager in a separate thread.
     *
     * @param httpClient httpClient
     */
    public static void shutdown(final HttpClient httpClient) {
        if (httpClient != null) {
            Thread shutdownThread = new Thread() {
                @Override
                public void run() {
                    HttpConnectionManager cm = httpClient.getHttpConnectionManager();
                    if (cm instanceof MultiThreadedHttpConnectionManager) {
                        ((MultiThreadedHttpConnectionManager) cm).shutdown();
                    } else if (cm instanceof SimpleHttpConnectionManager) {
                        ((SimpleHttpConnectionManager) cm).shutdown();
                    }
                }
            };
            shutdownThread.setName("HttpConnectionManager Shutdown Thread");
            shutdownThread.setDaemon(true);
            shutdownThread.start();
        }
    }

}
