/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */

package com.day.cq.audit;

import org.apache.sling.api.resource.ResourceResolver;

import java.util.List;


/**
 * The <code>AuditLog</code> defines an interface to retrieve audit event
 * information from the respective audit log.
 * The audit log is path based: the content path in the repository is
 * used as a unique identifier to group all entries.
 * <p>
 * The audit framework is still subject to change and will be improved.
 *
 * Adding entries to the audit log can be done by sending a job event
 * with the job topic {@link AuditLogEvent#JOB_TOPIC} and the {@link AuditLogEvent} stored
 * in the property {@link AuditLogEvent#AUDIT_EVENT_PROPERTY} or by directly calling
 * {@link #add(AuditLogEntry)} or {@link #add(List)}.
 */
public interface AuditLog {

    /**
     * Returns the latest event for the path and category.
     * @param category the event category
     * @param path the path of the event
     * @return the latest event with the given path or <code>null</code>.
     * throws {@link IllegalArgumentException} If category or path are null.
     * @deprecated since 6.3, use {@link #getLatestEvent(org.apache.sling.api.resource.ResourceResolver, String, String)} instead
     */
    @Deprecated
    AuditLogEntry getLatestEvent(String category, String path);

    /**
     * Returns the latest event for the path and category.
     * @param resolver ResourceResolver instance
     * @param category the event category
     * @param path the path of the event
     * @return the latest event with the given path or <code>null</code>.
     * throws {@link IllegalArgumentException} If category or path are null.
     */
    AuditLogEntry getLatestEvent(ResourceResolver resolver, String category, String path);

    /**
     * Returns the latest event for the path, type and category.
     * @param category the event category
     * @param path the path of the event
     * @param type the type of the event
     * @return the latest event with the given path or <code>null</code>.
     * throws {@link IllegalArgumentException} If category, type or path are null.
     * @deprecated since 6.3, use {@link #getLatestEvent(org.apache.sling.api.resource.ResourceResolver, String, String, String)} instead
     */
    @Deprecated
    AuditLogEntry getLatestEvent(String category, String path, String type);

    /**
     * Returns the latest event for the path, type and category.
     * @param resolver ResourceResolver instance
     * @param category the event category
     * @param path the path of the event
     * @param type the type of the event
     * @return the latest event with the given path or <code>null</code>.
     * throws {@link IllegalArgumentException} If category, type or path are null.
     */
    AuditLogEntry getLatestEvent(ResourceResolver resolver, String category, String path, String type);

    /**
     * Returns the latest events for the path and category.
     * @param category the event category
     * @param path the path of the events
     * @param max the maximum number of events to return or -1 for all.
     * @return an array of AuditEvents
     * throws {@link IllegalArgumentException} If category or path are null.
     * @deprecated since 6.3, use {@link #getLatestEvents(org.apache.sling.api.resource.ResourceResolver, String, String, int)} instead
     */
    @Deprecated
    AuditLogEntry[] getLatestEvents(String category, String path, int max);

    /**
     * Returns the latest events for the path and category.
     * @param resolver ResourceResolver instance
     * @param category the event category
     * @param path the path of the events
     * @param max the maximum number of events to return or -1 for all.
     * @return an array of AuditEvents
     * throws {@link IllegalArgumentException} If category or path are null.
     */
    AuditLogEntry[] getLatestEvents(ResourceResolver resolver, String category, String path, int max);

    /**
     * Returns the latest events for the path and category.
     * @param categories the event categories
     * @param path the path of the events
     * @param max the maximum number of events to return or -1 for all.
     * @return an array of AuditEvents
     * throws {@link IllegalArgumentException} If category or path are null.
     * @deprecated since 6.3, use {@link #getLatestEvents(org.apache.sling.api.resource.ResourceResolver, String[], String, int)} instead
     */
    @Deprecated
    AuditLogEntry[] getLatestEvents(String[] categories, String path, int max);

    /**
     * Returns the latest events for the path and category.
     * @param resolver ResourceResolver instance
     * @param categories the event categories
     * @param path the path of the events
     * @param max the maximum number of events to return or -1 for all.
     * @return an array of AuditEvents
     * throws {@link IllegalArgumentException} If category or path are null.
     */
    AuditLogEntry[] getLatestEvents(ResourceResolver resolver, String[] categories, String path, int max);

    /**
     * Return an array containing all audit log categories (that have entries)
     * @return An array with the categories or <code>null</code>
     */
    String[] getCategories();

    /**
     * Returns the latest events for all paths which will start
     * with the <code>prefix</code>.
     *
     * @param categories the event categories
     * @param pathPrefix the pattern to match
     * @param max the maximum number of events per path to return or -1 for all.
     * @return an array of AuditEvents
     * throws {@link IllegalArgumentException} If category or pathPrefix are null.
     * @deprecated since 6.3, use {@link #getLatestEventsFromTree(org.apache.sling.api.resource.ResourceResolver, String[], String, int)} instead
     */
    @Deprecated
    AuditLogEntry[] getLatestEventsFromTree(String[] categories, String pathPrefix, int max);

    /**
     * Returns the latest events for all paths which will start
     * with the <code>prefix</code>.
     *
     * @param resolver ResourceResolver instance
     * @param categories the event categories
     * @param pathPrefix the pattern to match
     * @param max the maximum number of events per path to return or -1 for all.
     * @return an array of AuditEvents
     * throws {@link IllegalArgumentException} If category or pathPrefix are null.
     */
    AuditLogEntry[] getLatestEventsFromTree(ResourceResolver resolver, String[] categories, String pathPrefix, int max);

    /**
     * Add a new audit log entry.
     * @param entry The audit log entry.
     */
    void add(AuditLogEntry entry);

    /**
     * Add new audit log entries
     * @param entries The audit log entries.
     */
    void add(List<AuditLogEntry> entries);

    /**
     * Move audit log entries from one place to another.
     */
    void move(String srcPath, String destPath);
}
