/*
 * Copyright 1997-2010 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.commons;

import javax.servlet.http.HttpServletRequest;

import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.ResourceResolver;

/**
 * Service interface to externalize URLs.
 * 
 * <p>
 * Allows creation of absolute URLs (i.e. including scheme and host) used
 * <b>outside the context of HTML</b> (where all URLs should be relative). A
 * typical example would be an RSS feed link:
 * <code>http://server.com/feed.xml</code>. Since instances itself cannot know
 * their externally visible URL if they are running behind a web layer, and
 * sometimes those links have to be created outside of the request scope, this
 * service provides a central place to configure those external URLs and build
 * them.
 * 
 * <h3>Details</h3>
 * 
 * <p>
 * There are the following options:
 * <ol>
 * <li>create an absolute URL to any configured server identified by a simple
 * name - based on configuration via the
 * {@link #externalLink(ResourceResolver, String, String, String) externalLink},
 * &nbsp;{@link #publishLink(ResourceResolver, String, String) publishLink} and
 * {@link #authorLink(ResourceResolver, String, String) authorLink} methods; the
 * domain can be one of these:
 * <ul>
 * <li>{@link #PUBLISH} "publish" - the public facing website DNS</li>
 * <li>{@link #AUTHOR} "author" - the authoring system DNS</li>
 * <li>{@link #LOCAL} "local" - the local instance</li>
 * <li>or any custom configured domain</li>
 * </ul>
 * </li>
 * <li>create an absolute URL based on the host dynamically provided in the
 * current request with
 * {@link #absoluteLink(SlingHttpServletRequest, String, String) absoluteLink}
 * (not recommended for most use cases)</li>
 * <li>create a relative URL that is relative to the current request using
 * {@link #relativeLink(SlingHttpServletRequest, String) relativeLink} (provided
 * for convenience, same as
 * {@link ResourceResolver#map(javax.servlet.http.HttpServletRequest, String)
 * ResourceResolver#map})</li>
 * </ol>
 * 
 * <p>
 * These elements are considered:
 * <ul>
 * <li><b>host name</b></li>
 * <li><b>port</b> (80 for http and 443 for https will be skipped)</li>
 * <li><b>context path</b> of the sling webapp</li>
 * <li><b>scheme</b> - usually defined by the caller (e.g. if it creates an
 * "ftp" or "webcal" link), but the configuration should contain a default
 * scheme (http or https) to indicate whether SSL is used by default or not for
 * normal http links; in that case use one of the methods without a scheme
 * argument</li>
 * <li><b>sling mappings</b> - done by passing a {@link ResourceResolver}; also
 * includes namespace mangling (<code>jcr:content</code> becomes
 * <code>_jcr_content)</code></li>
 * </ul>
 * 
 * <h3>Examples</h3>
 * 
 * <p>
 * Can be retrieved as a normal OSGi service:
 * 
 * <pre>
 * {@code @Reference}
 * Externalizer externalizer;
 * </pre>
 * 
 * It can also be adapted from a {@link ResourceResolver}:
 * 
 * <pre>
 * Externalizer externalizer = resourceResolver.adaptTo(Externalizer.class);
 * </pre>
 * 
 * <p>
 * Always pass a raw resource path. It might contain an extension, URL query
 * and/or fragment part already, but it is best to add those after the call:
 * 
 * <pre>
 * // public facing website
 * externalizer.publishLink(resolver, "/my/page") + ".html";
 * // => "http://www.website.com/contextpath/my/page.html"
 * 
 * externalizer.publishLink(resolver, "webcal", "/my/cal") + ".ics";
 * // => "webcal://www.website.com/contextpath/my/cal.ics"
 * 
 * externalizer.publishLink(resolver, "/my/path?query=part#fragment");
 * // => "http://www.website.com/contextpath/my/path?query=part#fragment"
 * 
 * // link to author
 * externalizer.authorLink(resolver, "/my/page") + ".html";
 * // => "http://author.website.com/contextpath/my/page.html"
 * 
 * // direct link to instance itself
 * externalizer.externalLink(resolver, Externalizer.LOCAL, "/my/page") + ".html";
 * // => "http://publish-3.internal/contextpath/my/page.html"
 * 
 * // custom configured domain
 * externalizer.externalLink(resolver, "mydomain", "/my/page") + ".html";
 * // => "http://mydomain.com/my/page.html"
 * 
 * // absolute link based on the request
 * externalizer.absoluteLink(slingRequest, "http", "/my/path");
 * // => "http://host.com/contextpath/my/path"
 * 
 * // relative links always require the request object 
 * externalizer.relativeLink(slingRequest, "/my/path");
 * // => "/contextpath/my/path"
 * </pre>
 * 
 * <h3>Note about the link checker for HTML</h3>
 * 
 * <p>
 * For basic <code>&lt;a&gt;</code> and <code>&lt;area&gt;</code> links in HTML,
 * the CQ link checker will automatically run
 * {@link ResourceResolver#map(javax.servlet.http.HttpServletRequest, String)
 * ResourceResolver#map(request, path)} to handle mappings, the context path and
 * namespace mangling. The same applies to <code>href</code>, <code>src</code>
 * and <code>action</code> attributes on any HTML element. <b>For those HTML
 * cases this utility should not be used</b>, as no absolute links should be
 * created in the context of HTML. If relative URLs need to be written in places
 * that are not covered by the link checker, such as generated Javascript or
 * CSS, use
 * {@link ResourceResolver#map(javax.servlet.http.HttpServletRequest, String)
 * ResourceResolver#map(request, path)} manually (or
 * {@link #relativeLink(SlingHttpServletRequest, String) relativeLink}, which is
 * the same).
 * 
 * <p>
 * However, any link that was already sent through this utility should also go
 * through untouched by the link checker an additional time (but only seen as
 * valid if the resource exists).
 */
public interface Externalizer {

    /**
     * Name for domain configuration that contains the instance's local address.
     * For example <code>http://author-1.internal:4502</code> or
     * <code>http://publish-3.internal:4503</code>.
     */ 
    final static String LOCAL = "local";

    /**
     * Name for domain configuration that contains the author DNS address. For
     * example <code>http://author.website.com</code>.
     */ 
    final static String AUTHOR = "author";

    /**
     * Name for domain configuration that contains the public website DNS
     * address. For example <code>http://www.website.com</code>.
     */ 
    final static String PUBLISH = "publish";

    /**
     * Creates an absolute URL for a named domain. Uses the configured default
     * scheme of that domain, or "http".
     * 
     * <p>
     * Use the standard {@link #LOCAL}, {@link #PUBLISH} or {@link #AUTHOR}
     * domains. Custom ones are also possible.
     * 
     * @see #publishLink(ResourceResolver, String)
     * @see #authorLink(ResourceResolver, String)
     * 
     * @since 5.5
     * 
     * @param resolver
     *            a resource resolver for handling the sling mappings and
     *            namespace mangling; can be <code>null</code>
     * @param domain
     *            name of the domain configuration to use
     * @param path
     *            a resource path; might contain extension, query or fragment,
     *            but plain paths are recommended; has to be without context
     *            path
     * @return an absolute URL string
     */
    String externalLink(ResourceResolver resolver, String domain, String path);

    /**
     * Creates an absolute URL for a named domain. Uses the scheme passed as
     * argument.
     * 
     * <p>
     * Use the standard {@link #LOCAL}, {@link #PUBLISH} or {@link #AUTHOR}
     * domains. Custom ones are also possible.
     * 
     * @see #publishLink(ResourceResolver, String, String)
     * @see #authorLink(ResourceResolver, String, String)
     * 
     * @since 5.5
     * 
     * @param resolver
     *            a resource resolver for handling the sling mappings and
     *            namespace mangling; can be <code>null</code>
     * @param domain
     *            name of the domain configuration to use
     * @param scheme
     *            a protocol scheme such as "http", that will be part of the URL
     * @param path
     *            a resource path; might contain extension, query or fragment,
     *            but plain paths are recommended; has to be without context
     *            path
     * @return an absolute URL string
     */
    String externalLink(ResourceResolver resolver, String domain, String scheme, String path);
    
    /**
     * Creates an absolute URL for the public website. Uses the configured
     * default scheme of that domain, or "http".
     * 
     * <p>
     * Shortcut for {@link #externalLink(ResourceResolver, String, String)
     * externalLink(resolver, Externalizer.PUBLISH, path)}.
     * 
     * @see #externalLink(ResourceResolver, String, String)
     * 
     * @since 5.5
     * 
     * @param resolver
     *            a resource resolver for handling the sling mappings and
     *            namespace mangling; can be <code>null</code>
     * @param path
     *            a resource path; might contain extension, query or fragment,
     *            but plain paths are recommended; has to be without context
     *            path
     * @return an absolute URL string
     */
    String publishLink(ResourceResolver resolver, String path);

    /**
     * Creates an absolute URL for the public website. Uses the scheme passed as
     * argument.
     * 
     * <p>
     * Shortcut for {@link #externalLink(ResourceResolver, String, String, String)
     * externalLink(resolver, Externalizer.PUBLISH, scheme, path)}.
     * 
     * @see #externalLink(ResourceResolver, String, String, String)
     * 
     * @since 5.5
     * 
     * @param resolver
     *            a resource resolver for handling the sling mappings and
     *            namespace mangling; can be <code>null</code>
     * @param scheme
     *            a protocol scheme such as "http", that will be part of the URL
     * @param path
     *            a resource path; might contain extension, query or fragment,
     *            but plain paths are recommended; has to be without context
     *            path
     * @return an absolute URL string
     */
    String publishLink(ResourceResolver resolver, String scheme, String path);

    /**
     * Creates an absolute URL for the authoring system. Uses the configured
     * default scheme of that domain, or "http".
     * 
     * <p>
     * Shortcut for {@link #externalLink(ResourceResolver, String, String)
     * externalLink(resolver, Externalizer.AUTHOR, path)}.
     * 
     * @see #externalLink(ResourceResolver, String, String)
     * 
     * @since 5.5
     * 
     * @param resolver
     *            a resource resolver for handling the sling mappings and
     *            namespace mangling; can be <code>null</code>
     * @param path
     *            a resource path; might contain extension, query or fragment,
     *            but plain paths are recommended; has to be without context
     *            path
     * @return an absolute URL string
     */
    String authorLink(ResourceResolver resolver, String path);
    
    /**
     * Creates an absolute URL for the authoring system. Uses the scheme passed as
     * argument.
     * 
     * <p>
     * Shortcut for {@link #externalLink(ResourceResolver, String, String, String)
     * externalLink(resolver, Externalizer.AUTHOR, scheme, path)}.
     * 
     * @see #externalLink(ResourceResolver, String, String, String)
     * 
     * @since 5.5
     * 
     * @param resolver
     *            a resource resolver for handling the sling mappings and
     *            namespace mangling; can be <code>null</code>
     * @param scheme
     *            a protocol scheme such as "http", that will be part of the URL
     * @param path
     *            a resource path; might contain extension, query or fragment,
     *            but plain paths are recommended; has to be without context
     *            path
     * @return an absolute URL string
     */
    String authorLink(ResourceResolver resolver, String scheme, String path);
        
    /**
     * Externalizes the given resource path relative to the URL of the
     * request.
     * 
     * <p>
     * <b>Note:</b> This is exactly the same as
     * <code>request.getResourceResolver().map(request, path)</code>.
     * 
     * <p>
     * Note that the result might be an absolute URL if the sling mappings
     * define an explicit hostname and the current request's hostname is
     * different.
     * 
     * @param request
     *            a sling http request object (required for context path and
     *            sling resource resolver mapping)
     * @param path
     *            a resource path; might contain extension, query or fragment,
     *            but plain paths are recommended; has to be without context
     *            path
     * @return an fully qualified URL string that is relative to the given
     *         request; it might be an absolute URL if the resource path is
     *         mapped to a different host than the current request
     */
    String relativeLink(SlingHttpServletRequest request, String path);

    /**
     * Externalizes the given resource path as an absolute URL based on the
     * request. The hostname (and port) are taken from the resource resolver
     * mapping configuration, if present, or <b>dynamically from the current
     * request</b> using {@link HttpServletRequest#getServerName()} and
     * {@link HttpServletRequest#getServerPort()}, while the scheme is given as
     * argument.
     * 
     * <p>
     * Use with care, as this is request dependent: the host header might not be
     * what is expected.
     * 
     * @param request
     *            a sling http request object (required for host, port, context
     *            path and sling resource resolver mapping)
     * @param scheme
     *            a protocol scheme such as "http", that will be part of the URL
     * @param path
     *            a resource path; might contain extension, query or fragment,
     *            but plain paths are recommended; has to be without context
     *            path
     * @return an absolute URL string
     */
    String absoluteLink(SlingHttpServletRequest request, String scheme, String path);

    /**
     * Externalizes the given resource path as an absolute URL. The hostname
     * (and port) are taken from the resource resolver mapping configuration, if
     * present, or from the service configuration. The context path will always
     * be taken from the service configuration (the service implementation is
     * free to use other mechanisms instead of configuration).
     * 
     * @deprecated since 5.5, use
     *             {@link #externalLink(ResourceResolver, String, String, String)
     *             externalLink(resolver, Externalizer.LOCAL, scheme, path)}
     *             instead
     * 
     * @param resolver
     *            a resource resolver for retrieving the sling mapping
     *            configuration; can be <code>null</code> to rely solely on this
     *            service's configuration of host and context path
     * @param scheme
     *            a protocol scheme such as "http", that will be part of the URL
     * @param path
     *            a resource path; might contain extension, query or fragment,
     *            but plain paths are recommended; has to be without context
     *            path
     * @return an absolute URL string
     */
    @Deprecated
    String absoluteLink(ResourceResolver resolver, String scheme, String path);

    /**
     * Externalizes the given resource path as an absolute URL. The hostname
     * (and port) as well as the context path are taken from the service
     * configuration (the service implementation is free to use other mechanisms
     * instead of configuration).
     * 
     * @deprecated since 5.5, use
     *             {@link #externalLink(ResourceResolver, String, String, String)
     *             externalLink(null, Externalizer.LOCAL, scheme, path)} instead
     * 
     * @param scheme
     *            a protocol scheme such as "http", that will be part of the URL
     * @param path
     *            a resource path; might contain extension, query or fragment,
     *            but plain paths are recommended; has to be without context
     *            path
     * @return an absolute URL string
     */
    @Deprecated
    String absoluteLink(String scheme, String path);
}
