/*
 * Copyright 1997-2010 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.commons;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.util.Collection;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.sling.commons.json.JSONArray;
import org.apache.sling.commons.json.JSONException;
import org.apache.sling.commons.json.JSONObject;
import org.apache.sling.commons.json.jcr.JsonItemWriter;

/**
 * Utility class that provides static methods for properly writing Javascript
 * and JSON snippets.
 */
public class JS {

    /**
     * Returns a Javascript string in string representation, including the
     * surrounding double-quotes. For example: <code>"foobar"</code>.
     *
     * <p>
     * If the string is null, <code>null</code> (as a Javascript null-value)
     * will be returned.
     * <p>
     * The Javascript string will be properly escaped.
     *
     * <p>
     * This is the same as
     * <code>same behaviour as JSONObject.valueToString(text)</code>.
     *
     * @param text
     *            a Java string
     * @return a qualified Javascript string literal or the string
     *         <code>null</code>
     */
    public static String str(String text) {
        try {
            return JSONObject.valueToString(text);
        } catch (JSONException e) {
            // this can never happen: when passing a String,
            // no exception can be thrown in valueToString
            return "null";
        }
    }

    /**
     * Returns a Javascript/JSON array in string representation from a Java
     * string array.
     *
     * <p>
     * Output will have no indentation and no new lines, but a single whitespace
     * between elements: <code>["one", "two", "three"]</code>
     *
     * <p>
     * If the given array is <code>null</code>, an empty array will be returned
     * (<code>[]</code>). The Javascript strings will be properly escaped.
     * {@link #str(String)} is used for each element of the array.
     *
     * @param array
     *            a string array
     * @return a qualified string representation of a Javascript array literal
     */
    public static String array(String[] array) {
        if (array == null) {
            return "[]";
        }
        StringBuilder builder = new StringBuilder();
        builder.append('[');
        for (int i = 0; i < array.length; i++) {
            builder.append( str(array[i]) );
            if (i < array.length-1) {
                builder.append(", ");
            }
        }
        builder.append(']');
        return builder.toString();
    }

    /**
     * Returns a Javascript/JSON array from a Java object collection. Same as
     * <code>new {@link JSONArray}(list).toString();</code> and behaves like
     * {@link #array(String[])}.
     *
     * @param list
     *            a collection of objects
     * @return a qualified string representation of a Javascript array literal
     */
    public static String array(Collection<?> list) {
        return new JSONArray(list).toString();
    }

    /**
     * Writes the given JCR Node and its entire subtree as JSON into the given
     * writer. Use {@link #writeNode(Writer, Node, int)} for controlling the
     * node depth.
     *
     * <p>
     * This is the same JSON as returned by Sling's default JSON renderer. Can
     * be used in JSPs to inline JSON for javascript code, for example:
     *
     * <pre>
     * var data = &lt;% JS.writeNode(out, currentNode); %&gt;;
     * </pre>
     *
     * which might result in:
     *
     * <pre>
     * var data = { &quot;jcr:primaryType&quot;: &quot;nt:unstructured&quot;, &quot;property&quot; : &quot;value&quot; };
     * </pre>
     *
     * <p>
     * This writes directly to the (JSP) output stream and thus needs to flush
     * it. Because of this it also works good for larger trees. To return it as
     * a string, use {@link #object(Node)}. Any exception will be passed to the
     * caller.
     *
     * @param out
     *            a writer, such as a {@link javax.servlet.jsp.JspWriter}, to write the JSON into.
     *            Will automatically be flushed before and after.
     * @param node
     *            the node to write
     * @throws IOException
     *             if there was a problem with the writer
     * @throws RepositoryException
     *             if some jcr error happened
     * @throws JSONException
     *             if writing the json failed
     */
    public static void writeNode(Writer out, Node node) throws IOException, RepositoryException, JSONException {
        writeNode(out, node, -1);
    }

    /**
     * Writes the given JCR Node and its subtree up until a given depth as JSON
     * into the given writer.
     *
     * <p>
     * This is the same JSON as returned by Sling's default JSON renderer. Can
     * be used in JSPs to inline JSON for javascript code, for example:
     *
     * <pre>
     * var data = &lt;% JS.writeNode(out, currentNode, 0); %&gt;;
     * </pre>
     *
     * which might result in:
     *
     * <pre>
     * var data = { &quot;jcr:primaryType&quot;: &quot;nt:unstructured&quot;, &quot;property&quot; : &quot;value&quot; };
     * </pre>
     *
     * <p>
     * This writes directly to the (JSP) output stream and thus needs to flush
     * it. Because of this it also works good for larger trees. To return it as
     * a string, use {@link #object(Node, int)}. Any exception will be passed to the
     * caller.
     *
     * @param out
     *            a writer, such as a {@link javax.servlet.jsp.JspWriter}, to write the JSON into.
     *            Will automatically be flushed before and after.
     * @param node
     *            the node to write
     * @param depth
     *            until which depth the tree should be written; 0 means the
     *            current node and its properties only; -1 means the whole tree.
     * @throws IOException
     *             if there was a problem with the writer
     * @throws RepositoryException
     *             if some jcr error happened
     * @throws JSONException
     *             if writing the json failed
     */
    public static void writeNode(Writer out, Node node, int depth) throws IOException, RepositoryException, JSONException {
        out.flush();

        JsonItemWriter jsonWriter = new JsonItemWriter(null);
        jsonWriter.dump(node, new PrintWriter(out), depth);

        out.flush();
    }

    /**
     * Returns a given JCR Node and its entire subtree as JSON object. Use
     * {@link #object(Node, int)} for controlling the node depth.
     *
     * <p>
     * This is the same JSON as returned by Sling's default JSON renderer. Can
     * be used in JSPs to inline JSON for javascript code, for example:
     *
     * <pre>
     * var data = &lt;%= JS.object(currentNode) %&gt;;
     * </pre>
     *
     * which might result in:
     *
     * <pre>
     * var data = { &quot;jcr:primaryType&quot;: &quot;nt:unstructured&quot;, &quot;property&quot; : &quot;value&quot; };
     * </pre>
     *
     * <p>
     * For larger node trees it is probably more efficient to stream it by using
     * {@link #writeNode(Writer, Node)}. Any exception will be passed to the
     * caller.
     *
     * @param node
     *            the node to write
     * @throws RepositoryException
     *             if some jcr error happened
     * @throws JSONException
     *             if writing the json failed
     * @return a JSON object as string representation
     */
    public static String object(Node node) throws RepositoryException, JSONException {
        return object(node, -1);
    }

    /**
     * Returns a given JCR Node and its subtree up until a given depth as JSON.
     *
     * <p>
     * This is the same JSON as returned by Sling's default JSON renderer. Can
     * be used in JSPs to inline JSON for javascript code, for example:
     *
     * <pre>
     * var data = &lt;%= JS.object(currentNode) %&gt;;
     * </pre>
     *
     * which might result in:
     *
     * <pre>
     * var data = { &quot;jcr:primaryType&quot;: &quot;nt:unstructured&quot;, &quot;property&quot; : &quot;value&quot; };
     * </pre>
     *
     * <p>
     * For larger node trees it is probably more efficient to stream it by using
     * {@link #writeNode(Writer, Node, int)}. Any exception will be passed to
     * the caller.
     *
     * @param node
     *            the node to write
     * @param depth
     *            until which depth the tree should be written; 0 means the
     *            current node and its properties only; -1 means the whole tree.
     * @throws RepositoryException
     *             if some jcr error happened
     * @throws JSONException
     *             if writing the json failed
     * @return a JSON object as string representation
     */
    public static String object(Node node, int depth) throws RepositoryException, JSONException {
        JsonItemWriter jsonWriter = new JsonItemWriter(null);
        StringWriter out = new StringWriter();
        jsonWriter.dump(node, out, depth);
        return out.toString();
    }
}
