/*
 * Copyright 1997-2011 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */

package com.day.cq.commons;

import com.day.cq.xss.ProtectionContext;
import com.day.cq.xss.XSSProtectionException;
import com.day.cq.xss.XSSProtectionService;
import org.apache.sling.commons.json.JSONException;
import org.apache.sling.commons.json.JSONObject;
import org.apache.sling.commons.json.io.JSONWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * JSON writer utilities
 * <p/>
 * Enables to specify a write mode while adding a JSON property.
 * According to the provided write mode, the property value may be encoded
 * through the XSS protection service
 *
 * @since 5.4
 */
public class JSONWriterUtil {

    private static final Logger log = LoggerFactory.getLogger(JSONWriterUtil.class);

    /**
     * Key suffix for XSS protected properties
     */
    public static final String KEY_SUFFIX_XSS = "_xss";

    /**
     * JSON write modes
     */
    public enum WriteMode {
        PLAIN_TEXT,
        AVOID_XSS,
        BOTH
    }

    /**
     * Puts a value into a JSON object according to provided write mode.
     *
     * @param object    JSON object
     * @param key       Key to write
     * @param value     Value to write
     * @param writeMode Write mode
     * @param xss       XSS protection service
     *
     * @throws JSONException If value could not be put into the object
     */
    public static void put(JSONObject object, String key, String value, WriteMode writeMode, XSSProtectionService xss) throws JSONException {
        Item item = new Item(key, value, writeMode, xss, log);

        if (item.isPlainText()) {
            // Plain text value
            object.put(item.getKey(), item.getValue());
        }

        if (item.isAvoidXSS() && null != xss) {
            object.put(item.getXssKey(), item.getXssValue());
        }
    }

    /**
     * Puts an optional value into a JSON object according to provided write mode.
     *
     * @param object    JSON object
     * @param key       Key to write
     * @param value     Value to write
     * @param writeMode Write mode
     * @param xss       XSS protection service
     *
     * @throws JSONException If value could not be put into the object
     */
    public static void putOptional(JSONObject object, String key, String value, WriteMode writeMode, XSSProtectionService xss) throws JSONException {
        if (value != null) {
            put(object, key, value, writeMode, xss);
        }
    }

    /**
     * Writes a value to JSON writer according to provided write mode.
     *
     * @param writer    JSON writer
     * @param key       Key to write
     * @param value     Value to write
     * @param writeMode Write mode
     * @param xss       XSS protection service
     * @throws JSONException If value could not be written
     */
    public static void write(JSONWriter writer, String key, String value, WriteMode writeMode, XSSProtectionService xss) throws JSONException {
        Item item = new Item(key, value, writeMode, xss, log);

        if (item.isPlainText()) {
            // Plain text value
            writer.key(item.getKey()).value(item.getValue());
        }

        if (item.isAvoidXSS() && null != xss) {
            writer.key(item.getXssKey()).value(item.getXssValue());
        }
    }

    /**
     * Writes an optional value to JSON writer according to provided write mode.
     *
     * @param writer    JSON writer
     * @param key       Key to write
     * @param value     Value to write
     * @param writeMode Write mode
     * @param xss       XSS protection service
     * @throws JSONException If value could not be written
     */
    public static void writeOptional(JSONWriter writer, String key, String value, WriteMode writeMode, XSSProtectionService xss) throws JSONException {
        if (value != null) {
            write(writer, key, value, writeMode, xss);
        }
    }

}

class Item {

    private final String key;
    private final String value;
    private final JSONWriterUtil.WriteMode writeMode;
    private final XSSProtectionService xss;
    private final Logger log;
    private String xssKey;
    private String xssValue;

    Item(String key, String value, JSONWriterUtil.WriteMode writeMode, XSSProtectionService xss, Logger log) throws JSONException {
        this.key = key;
        this.value = value;
        this.writeMode = writeMode;
        this.xss = xss;
        this.log = log;
        init();
    }

    private void init() throws JSONException {
        if (isAvoidXSS()) {
            // Write XSS protected value
            if (xss != null) {
                xssKey = key;
                if (isPlainText()) {
                    xssKey += JSONWriterUtil.KEY_SUFFIX_XSS;
                }
                try {
                    xssValue = xss.protectForContext(ProtectionContext.PLAIN_HTML_CONTENT, value);
                } catch (XSSProtectionException e) {
                    throw new JSONException("Unable to write XSS protected value for key[" + key + "] " + value, e);
                }
            } else {
                log.warn("XSS protection service was not provided or not available while processing key [{}]", key);
            }
        }
    }

    public String getKey() {
        return key;
    }

    public String getValue() {
        return value;
    }

    public String getXssKey() {
        return xssKey;
    }

    public String getXssValue() {
        return xssValue;
    }

    /**
     * Check if provided write mode includes 'Plain text'
     *
     * @return <code>true</code> if the write mode includes 'Plain text'
     */
    public boolean isPlainText() {
        return writeMode == null || JSONWriterUtil.WriteMode.PLAIN_TEXT.equals(writeMode) || JSONWriterUtil.WriteMode.BOTH.equals(writeMode);
    }

    /**
     * Check if provided write mode includes 'Avoid XSS'
     *
     * @return <code>true</code> if the write mode includes 'Avoid XSS'
     */
    public boolean isAvoidXSS() {
        return JSONWriterUtil.WriteMode.AVOID_XSS.equals(writeMode) || JSONWriterUtil.WriteMode.BOTH.equals(writeMode);
    }

}


