/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.commons;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import com.day.cq.commons.jcr.JcrConstants;

/**
 * Implements a labeled resource that reads the values from jcr properties.
 */
public class JcrLabeledResource implements LabeledResource {

    /**
     * internal path
     */
    private final String path;

    /**
     * internal name
     */
    private final String name;

    /**
     * internal title
     */
    private final String title;

    /**
     * internal description
     */
    private final String description;

    /**
     * Creates a new JcrLabeledResource and initializes its field from the
     * properties of the given node or it's <code>jcr:content</code> child node.
     *
     * @param node the node
     * @throws RepositoryException if an error during repository access occurs.
     */
    public JcrLabeledResource(Node node) throws RepositoryException {
        path = node.getPath();
        name = node.getName();
        Node contentNode = node.hasNode(JcrConstants.JCR_CONTENT)
                ? node.getNode(JcrConstants.JCR_CONTENT)
                : null;
        // get title
        if (node.hasProperty(JcrConstants.JCR_TITLE)) {
            title = node.getProperty(JcrConstants.JCR_TITLE).getString();
        } else if (contentNode != null && contentNode.hasProperty(JcrConstants.JCR_TITLE)) {
            title = contentNode.getProperty(JcrConstants.JCR_TITLE).getString();
        } else {
            title = null;
        }
        // get description
        if (node.hasProperty(JcrConstants.JCR_DESCRIPTION)) {
            description = node.getProperty(JcrConstants.JCR_DESCRIPTION).getString();
        } else if (contentNode != null && contentNode.hasProperty(JcrConstants.JCR_DESCRIPTION)) {
            description = contentNode.getProperty(JcrConstants.JCR_DESCRIPTION).getString();
        } else {
            description = null;
        }
    }

    /**
     * {@inheritDoc}
     *
     * @return the path of the underlying node
     */
    public String getPath() {
        return path;
    }

    /**
     * {@inheritDoc}
     *
     * @return the name of the underlying node
     */
    public String getName() {
        return name;
    }

    /**
     * {@inheritDoc}
     *
     * @return the value of the <code>jcr:title</code> property of the underlying
     *         node or of it's <code>jcr:content</code> child node if it exists.
     *         <code>null</code> otherwise.
     */
    public String getTitle() {
        return title;
    }

    /**
     * {@inheritDoc}
     *
     * @return the value of the <code>jcr:description</code> property of the underlying
     *         node or of it's <code>jcr:content</code> child node if it exists.
     *         <code>null</code> otherwise.
     */
    public String getDescription() {
        return description;
    }
}