/*
 * Copyright 1997-2010 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.commons;

import java.util.Locale;

/**
 * Implements a wrapper around a locale that supports different codes for
 * country and language. This is mostly used as workaround for language
 * trees that don't follow the same format for locales in java.util.Locale.
 */
public class Language {

    private final Locale locale;

    private final String language;

    private final String country;

    public Language(Locale locale) {
        this.locale = locale;
        language = locale.getLanguage();
        country = locale.getCountry();
    }

    public Language(String code) {
        // support both "-" and "_" as delimiters
        code = code.replaceAll("-", "_");

        int idx = code.indexOf('_');
        if (idx < 0) {
            language = code;
            country = "";
            locale = new Locale(language);
        } else {
            language = code.substring(0, idx);
            country = code.substring(idx + 1);
            locale = new Locale(language, country);
        }
    }

    public Language(Locale locale, String language, String country) {
        this.locale = locale;
        this.language = language;
        this.country = country;
    }

    public Language(String language, String country) {
        this.language = language;
        this.country = country;
        this.locale = new Locale(language, country);
    }

    public Locale getLocale() {
        return locale;
    }

    public String getLanguageCode() {
        return language;
    }

    public String getCountryCode() {
        return country;
    }

    public String toString() {
        final StringBuilder sb = new StringBuilder(language);
        if (country.length() > 0) {
            sb.append("_").append(country);
        }
        return sb.toString();
    }

    /**
     * Parses a language code such as "de", "de_CH", "de_DE_EURO" or "de-CH" and
     * returns the proper {@link Locale} object.
     */
    public static Locale getLocale(String code) {
        if (code == null) {
            return null;
        }
        // support both "-" and "_" as delimiters
        code = code.replaceAll("-", "_");

        // check for existence of country (de_CH)
        int idx = code.indexOf('_');
        if (idx < 0) {
            return new Locale(code);
        } else {
            String language = code.substring(0, idx);
            String country = code.substring(idx + 1);
            // check for existence of variant (de_DE_EURO)
            idx = country.indexOf('_');
            if (idx < 0) {
                return new Locale(language, country);
            } else {
                return new Locale(language, country.substring(0, idx), country.substring(idx + 1));
            }
        }
    }
}