/*
 * Copyright 1997-2010 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.commons.inherit;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceUtil;
import org.apache.sling.api.resource.ValueMap;

import com.day.cq.commons.ValueMapWrapper;
import com.day.cq.commons.jcr.JcrConstants;

/**
 * An {@link InheritanceValueMap} for a given {@link Resource} that will inherit values from
 * ancestral pages.
 *
 * <p>
 *     For example, given: <code>/content/parent/page/jcr:content/footer/image/@width</code>,
 *     the <code>HierarchyNodeInheritanceValueMap</code> will search for a <code>footer/image/@width</code>
 *     property in:
 *     <ul>
 *         <li><code>/content/parent/page/jcr:content/footer/image/@width</code></li>
 *         <li><code>/content/parent/jcr:content/footer/image/@width</code></li>
 *         <li><code>/content/jcr:content/footer/image/@width</code></li>
 *     </ul>
 *     Having not found it in any of those locations, it will then return <code>null</code>.
 *
 * <p>
 *     Note that <code>HierarchyNodeInheritanceValueMap</code> searches <b>only</b> the page
 *     hierarchy.  It will <b>not</b> (for instance), look in:
 *     <ul>
 *         <li><code>/content/parent/page/jcr:content/footer/@width</code></li>
 *     </ul>
 *     See {@link ComponentInheritanceValueMap} for that functionality.
 */
public class HierarchyNodeInheritanceValueMap extends ValueMapWrapper implements InheritanceValueMap {

    private Resource resource;

    /**
     * Will wrap the {@link ValueMap} returned by
     * <code>resource.adaptTo(ValueMap.class)</code>, or use an empty map if
     * <code>resource</code> is null or if it doesn't adapt to a ValueMap. The
     * inheritance is internally resolved by fetching the parent resource and
     * wrapping it into an {@link HierarchyNodeInheritanceValueMap} as well.
     * 
     * @param resource
     *            the resource to start from
     */
    public HierarchyNodeInheritanceValueMap(Resource resource) {
        super(ResourceUtil.getValueMap(resource));
        this.resource = resource;
    }

    /**
     * Use this if the underlying {@link Resource} for a {@link ValueMap} is not
     * available and no inheritance is needed. Using the inheritance-enabled
     * {@link #getInherited(String, Class) getter}
     * {@link #getInherited(String, Object) methods} will behave exactly like
     * the normal ValueMap getters.
     * 
     * @param map
     *            a ValueMap to wrap
     */
    public HierarchyNodeInheritanceValueMap(ValueMap map) {
        super(map);
        this.resource = null;
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T> T get(String name, Class<T> type) {
        // overwritten to fix NPE
        if (type == null) {
            return (T) get(name);
        }
        return super.get(name, type);
    }

    public <T> T getInherited(String name, Class<T> type) {
        T value = get(name, type);
        if (value == null) {
            value = getParentPageValue(getInnerPath(resource), name, type);
        }
        return value;
    }
    
    @SuppressWarnings("unchecked")
    public <T> T getInherited(String name, T defaultValue) {
        Class<T> type;
        if (defaultValue == null) {
            type = null;
        } else {
            // special handling in case the default value implements one
            // of the interface types supported by the convertToType method
            type = (Class<T>) defaultValue.getClass();
        }
        
        T value = getInherited(name, type);
        if (value == null) {
            value = defaultValue;
        }
        
        return value;
    }
    
    // --------------------------------------------------< protected >---

    protected <T> T getParentPageValue(final String innerPath, final String name, final Class<T> type) {
        if (resource == null) {
            return null;
        }
        
        // go up ancestors (allow empty paths)
        Resource parent = ResourceUtil.getParent(resource);
        boolean isContentNode = ResourceUtil.getName(resource).equals(JcrConstants.JCR_CONTENT);
        if (parent == null) {
            parent = getNextExistingParent(resource);
            if (parent == null) {
                // reached the root, but nothing was found
                return null;
            }
            isContentNode = false;
        }
        
        // if we aren't the jcr:content child node ourself...
        if (!isContentNode) {
            // ...check jcr:content subnode of any hierarchy node
            ResourceResolver resolver = parent.getResourceResolver();
            Resource content = resolver.getResource(parent, JcrConstants.JCR_CONTENT);
            if (content != null) {
                // look up innerPath
                Resource innerResource = resolver.getResource(content, innerPath);
                // get from that value map
                T value = ResourceUtil.getValueMap(innerResource).get(name, type);
                if (value != null) {
                    return value;
                }
            }
        }
        
        // now check the parent
        return new HierarchyNodeInheritanceValueMap(parent).getParentPageValue(innerPath, name, type);
    }
    
    /**
     * Returns the page local path (aka localstruct). For example:
     * /some/page/jcr:content/par/comp => par/comp
     */
    protected static String getInnerPath(Resource resource) {
        if (resource == null) {
            return ".";
        }
        final String resPath = resource.getPath();
        int pos = resPath.indexOf(JcrConstants.JCR_CONTENT + "/");
        if (pos <= 0) {
            return ".";
        }
        return resPath.substring(pos + JcrConstants.JCR_CONTENT.length() + 1);
    }
    
    /**
     * Returns the next existing ancestor. Normally this is the parent,
     * but if the resource (and one or more of its parents) does not exist,
     * this will walk up the path node by node to find the first existing
     * ancestor. Returns null for the parent of the root "/".
     */
    protected static Resource getNextExistingParent(Resource resource) {
        ResourceResolver resolver = resource.getResourceResolver();
        
        String path = resource.getPath();
        do {
            path = ResourceUtil.getParent(path);
            // reached root => stop
            if (path == null) {
                return null;
            }
            resource = resolver.getResource(path);
        } while (resource == null);
        
        return resource;
    }
}
