package com.day.cq.contentsync;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Date;
import java.util.Iterator;

import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.servlet.http.HttpServletResponse;

import aQute.bnd.annotation.ProviderType;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;

import com.day.cq.contentsync.config.Config;

/**
 * The <code>ContentSyncManager</code> listens for changes to content
 * sync configurations stored in the repository and updates
 * the zip content cache accordingly.
 */
@ProviderType
public interface ContentSyncManager {
	
	/**
	 * Update the cache for the given config resource and using
	 * the session for personalized content.
	 * @param resource	The config resource
	 * @param session	The session
	 */
	public void updateCache(Resource resource, Session session);
	
	/**
	 * Update the cache for the given config and using
	 * the session for personalized content.
     * If cached updated, create an update timestamp.
	 * @param config	The config
	 * @param session	The session
     * @return true if cache was filled, false otherwise
	 */
	public boolean updateCache(Config config, Session session);
	
     /**
     * Clears the cache for the given config and using
     * the session for personalized content.
     * @param config	The config
     * @param session	The session
     */
    public void clearCache(Config config, Session session);

	/**
	 * Checks if there are any updates for the given config
	 * by comparing to the last modified date.
	 * @param resource			The config's resource
	 * @param ifModifiedSince	The last modified date
	 * @return <code>true</code> when updates available, <code>false</code> otherwise
	 */
	public boolean hasUpdates(Resource resource, Date ifModifiedSince);

	/**
	 * Returns the latest update timestamp for the given
	 * config and user (for personalized configs).
	 * @param config The configuration
	 * @param session The session
	 * @return Latest update timestamp or <code>0</code> if none exists
	 */
	public Long getLatestTimestamp(Config config, Session session);

	/**
	 * This method builds a zip file according to the given config,
	 * last modified date and session and returns a URI for identification.
	 * Use {@link ContentSyncManager#sendZip} for actual delivery of the file.
	 * @param config			The config resource
	 * @param ifModifiedSince	The last modified date
	 * @param session			The session
	 * @return Path to zip file
	 */
	public String getZip(Resource config, Date ifModifiedSince, Session session) throws RepositoryException, IOException;
	
	/**
	 * Sets the correct content type and copies the zip file
	 * to the response's output stream.
	 * @param response		The response
	 * @param requestURI	Request URI of the zip file
	 */
	@Deprecated
	public void sendZip(HttpServletResponse response, String requestURI);

    /**
     * Sets the correct content type and copies the zip file
     * to the response's output stream.
     * @param response      The response
     * @param requestURI    Request URI of the zip file.
     */
	public void sendZip(Session session, OutputStream out, String requestURI);

    /**
     * Sets the correct content type and copies the zip file
     * to the response's output stream.
     * @param session       The session for reading the zip
     * @param response      The response
     * @param requestURI    Request URI of the zip file
     */
    public void sendZip(Session session, HttpServletResponse response, String requestURI);
    
    /**
     * Get the list of available configs.
     * @param resolver The resource resolver
     * @return The list of configs
     */
	@Deprecated
    public Iterator<Config> getConfigs();
	
	/**
	 * Get the list of available configs.
	 * @param resolver The resource resolver
	 * @return The list of configs
	 */
	public Iterator<Config> getConfigs(ResourceResolver resolver);
}