/*
 * Copyright 1997-2009 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.dam.api;

import org.apache.sling.api.resource.Resource;

import aQute.bnd.annotation.ProviderType;

import java.io.InputStream;
import java.util.Calendar;
import java.util.Collection;

import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;

/**
 * The <code>AssetManager</code> provides utility methods for assets.
 * <p/>
 * The asset manager can be retrieved as follows:
 * <blockquote><pre>
 *   ...
 *   AssetManager manager = resourceResolver.adaptTo(AssetManager.class);
 *   ...
 * </pre></blockquote>
 */

@ProviderType
public interface AssetManager {

    /**
     * Restores an <code>Asset<code>.
     *
     * @param revisionId revision id
     *
     * @return the restored {@link Asset}
     *
     * @throws Exception Upon encountering an error while restoring an asset.
     */
    Asset restore(String revisionId) throws Exception;

    /**
     * Lists all available {@link Revision}s.
     *
     * @param path path of asset
     * @param cal  starting date to search revisions or <code>null</code> in order to fetch all
     *
     * @return all available {@link Revision}s
     *
     * @throws Exception Upon encountering an error while getting the revisions for an asset.
     */
    Collection<Revision> getRevisions(String path, Calendar cal) throws Exception;

    /**
     * This method creates the complete asset structure in <i>/content/dam</i>. The following structure is created:
     * <xmp> + file.jpg (dam:Asset) + jcr:content )(dam:AssetContent) + renditions (sling:OrderedFolder) + metadata
     * (nt:unstructured)
     * <p/>
     * </xmp>
     * <p/>
     * The given <code>binaryPath</code> is translated into the final asset location (see {@link
     * com.day.cq.dam.commons.util.DamUtil#binaryToAssetPath(String)}).
     * <p/>
     *
     * @param binaryPath The path of the asset's binary in <i>/var/dam</i>.
     * @param doSave     Whether the repository changes are saved or not.
     *
     * @return The {@link javax.jcr.Node} representing the created asset.
     */
    Asset createAssetForBinary(String binaryPath, boolean doSave);

    /**
     * Returns the {@link Resource} of the {@link Asset} corresponding to the binary given with the <code>path</code>
     * parameter.
     * <blockquote><pre>
     *   DamUtil.getAssetNode("/var/dam/myfolder/test.jpg", session)    =  Resource("/content/dam/myfolder/test.jpg")
     * </pre></blockquote>
     *
     * @param binaryPath The path of the binary.
     *
     * @return The resource representing the binary's asset, or <code>null</code> if it couldn't be found.
     */
    Asset getAssetForBinary(String binaryPath);

    /**
     * Removes the {@link javax.jcr.Node} of the {@link Asset} corresponding to the binary given with the
     * <code>path</code> parameter.
     * <blockquote><pre>
     *   DamUtil.removeAssetNode("/var/dam/myfolder/test.jpg", session) => Delete Node("/content/dam/myfolder/test.jpg")
     * </pre></blockquote>
     *
     * @param binaryPath The path of the binary.
     *
     * @return <code>true</code> if the asset was successfully removed.
     */
    boolean removeAssetForBinary(String binaryPath);

    /**
     * Creates a new {@link Asset} at the given <code>path</code>. If an asset
     * already exists at the given <code>path</code>, its original rendition is
     * updated instead of creating a new asset. If inputStream is null new
     * {@link Asset} is created without original rendition. If an asset already
     * exists at given path and inputstream is null, original rendition is not
     * updated.
     *
     * @param path The path of the asset to be created.
     * @param is The input stream of the new asset's original binary.
     * @param mimeType The mime type of the new asset's original binary.
     * @param doSave Whether the repository changes are saved or not.
     * @return The newly created asset.
     */
    Asset createAsset(String path, InputStream is, String mimeType, boolean doSave);

    /**
     * Create a new {@link Revision} of the asset. The revision will be created as a standard JCR version of the
     * underlying asset node.
     *
     * @param asset   The asset to version
     * @param label   version label
     * @param comment version comment
     *
     * @return The created revision
     *
     * @throws Exception Thrown when an error during version creation occurred.
     */
    Revision createRevision(Asset asset, String label, String comment) throws Exception;

    /**
     * Ask Asset ID Provider (associated with Asset Manager) to assign ID
     * (if needed) to an asset and establish its parentage (dam:parentAssetID)
     *
     * @param asset   The asset to update
     *
     * @throws RepositoryException
     * @throws PathNotFoundException
     */
    String assignAssetID(Asset asset) throws PathNotFoundException, RepositoryException;
}
