/*
 * Copyright 1997-2009 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.dam.api.handler;

import com.adobe.granite.asset.api.AssetRelation;
import com.day.cq.dam.api.Asset;
import com.day.cq.dam.api.AssetHandlerException;
import com.day.cq.dam.api.Rendition;
import com.day.cq.dam.api.metadata.ExtractedMetadata;
import com.day.cq.dam.api.thumbnail.ThumbnailConfig;

import javax.jcr.Node;
import javax.jcr.Session;

import java.awt.Dimension;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * The <code>AssetHandler</code> inteface specifies the handling of assets, specifically during importing or updating
 * the binary of an asset.
 */
public interface AssetHandler {

    /**
     * Indicates if the handler supports processing of <code>sub assets</code>
     *
     * @return <code>true</code> if handler is able to process sub assets
     */
    boolean canHandleSubAssets();


    /**
     * Thumbnails of an asset are created during import/update of an asset and it's original binary. Currently this
     * method is used by the {@link com.day.cq.dam.core.process.CreateThumbnailProcess} and the {@link
     * com.day.cq.dam.core.process.CommandLineProcess} as part of the <i>DAM Update Asset</i> workflow. Implementations
     * may choose to create asset thumbnails for every entry in the <code>thumbnailConfigs</code> list argument, to not
     * create any thumbnails at all, or to create a generic thumbnail independent of the given config. Thumbnails
     * themselves are renditions of an asset and must be persisted as such (see {@link
     * com.day.cq.dam.api.Asset#addRendition(String, java.io.InputStream, String)}).
     * <p/>
     * The thumbnail renditions are created depending on the given configuration list:
     * <blockquote><pre>
     * {
     *     height:h1,
     *     width: w1,
     *     doCenter: true (or false)
     * },
     * {
     * ...
     * }
     * </pre></blockquote>
     * Sample:
     * <blockquote><pre>
     *   ...
     *   final ArrayList<Map<String, Object>> config = new ArrayList<Map<String, Object>>();
     * <p/>
     *   config.add(new HashMap<String, Object>() {{
     *       put("width", 140);
     *       put("height", 100);
     *       put("doCenter", true);
     *   }});
     *   config.add(new HashMap<String, Object>() {{
     *       put("width", 80);
     *       put("height", 80);
     *       put("doCenter", true);
     *   }});
     * <p/>
     *  handler.createThumbnailsExt(
     *           getTiff(),
     *           asset.adaptTo(Node.class).getNode("jcr:content/renditions"),
     *           session,
     *           config);
     *  ...
     * </pre></blockquote>
     *
     * @param asset   the asset/file for which the thumbnails are generated
     * @param configs thumbnail configurations
     * @throws IOException is thrown in case while an error occurred while fetching the image
     * @see com.day.cq.dam.api.Asset#addRendition(String, java.io.InputStream, String)
     * @see com.day.cq.dam.api.Asset#getRenditions()
     * @see com.day.cq.dam.api.Asset#getRendition(String) )
     * @see com.day.cq.dam.api.Asset#getRendition(com.day.cq.dam.api.RenditionPicker)
     * @since 5.4.0
     */
    void createThumbnails(Asset asset, Collection<ThumbnailConfig> configs) throws IOException;

    /**
     * Creates thumbnails for this asset. Which thumbnails are created is automatically derived from the existing
     * thumbnails (renditions with the <i>cq5dam.thumbnail</i> prefix). This essentially amounts to re-creating existing
     * thumbnails.
     *
     * @param asset The asset for which to create thumbnails.
     * @throws java.io.IOException If an error occurred while extracting the image.
     */
    void createThumbnails(Asset asset) throws IOException;

    /**
     * Creates thumbnails for an {@link Asset}s particular {@link Rendition}.
     *
     * @param asset     The asset for which to created thumbnails.
     * @param rendition The rendition serving as the thumbnail basis.
     * @param configs   The thumbnail configurations.
     * @throws IOException If an error occurred extracting an image from the rendition.
     */
    void createThumbnails(Asset asset, Rendition rendition, Collection<ThumbnailConfig> configs) throws IOException;

    /**
     * The thumbnail renditions are created depending on the dimensions list. The dimensions list has the following
     * format: <code>[width1,height1], [width2,height2], ... </code>
     *
     * @param asset           the asset/file for which the thumbnails are generated
     * @param renditionFolder the folder where the thumbnails belong to
     * @param session         the jcr session shich is used to store the thumbnails
     * @param dimensions      the dimensions for the "ordered" thumbnails
     * @deprecated Use {@link #createThumbnails(com.day.cq.dam.api.Asset, java.util.Collection)} instead.
     */
    void createThumbnails(Node asset, Node renditionFolder, Session session, List<Integer[]> dimensions);

    /**
     * Thumbnails of an asset are created during import/update of an asset and it's original binary. Currently this
     * method is used by the {@link com.day.cq.dam.core.process.CreateThumbnailProcess} and the {@link
     * com.day.cq.dam.core.process.CommandLineProcess} as part of the <i>DAM Update Asset</i> workflow. Implementations
     * may choose to create asset thumbnails for every entry in the <code>thumbnailConfigs</code> list argument, to not
     * create any thumbnails at all, or to create a generic thumbnail independent of the given config. Thumbnails
     * themselves are renditions of an asset and must be persisted as such (see {@link
     * com.day.cq.dam.api.Asset#addRendition(String, java.io.InputStream, String)}).
     * <p/>
     * The thumbnail renditions are created depending on the given configuration list:
     * <blockquote><pre>
     * {
     *     height:h1,
     *     width: w1,
     *     doCenter: true (or false)
     * },
     * {
     * ...
     * }
     * </pre></blockquote>
     * Sample:
     * <blockquote><pre>
     *   ...
     *   final ArrayList<Map<String, Object>> config = new ArrayList<Map<String, Object>>();
     * <p/>
     *   config.add(new HashMap<String, Object>() {{
     *       put("width", 140);
     *       put("height", 100);
     *       put("doCenter", true);
     *   }});
     *   config.add(new HashMap<String, Object>() {{
     *       put("width", 80);
     *       put("height", 80);
     *       put("doCenter", true);
     *   }});
     * <p/>
     *  handler.createThumbnailsExt(
     *           getTiff(),
     *           asset.adaptTo(Node.class).getNode("jcr:content/renditions"),
     *           session,
     *           config);
     *  ...
     * </pre></blockquote>
     *
     * @param asset            the asset/file for which the thumbnails are generated
     * @param renditionFolder  the folder where thumbnails (renditions) shall be stored
     * @param session          the jcr session shich is used to store the thumbnails
     * @param thumbnailConfigs thumbnail configurations
     * @throws IOException is thrown in case while an error occured while fetching the image
     * @see com.day.cq.dam.api.Asset#addRendition(String, java.io.InputStream, String)
     * @see com.day.cq.dam.api.Asset#getRenditions()
     * @see com.day.cq.dam.api.Asset#getRendition(String) )
     * @see com.day.cq.dam.api.Asset#getRendition(com.day.cq.dam.api.RenditionPicker)
     * @since 5.3.0
     * @deprecated Use {@link #createThumbnails(com.day.cq.dam.api.Asset, java.util.Collection)} instead.
     */
    void createThumbnailsExt(Node asset, Node renditionFolder,
                             Session session, List<Map<String, Object>> thumbnailConfigs)
            throws IOException;

    /**
     * This method is called when an asset is exported or the asset is downloaded.
     *
     * @param asset  the file node
     * @param stream the <code>OutputStream</code> where the exported asset is put.
     * @throws com.day.cq.dam.api.AssetHandlerException
     *          in case the export work failed.
     * @deprecated Use {@link #exportAsset(com.day.cq.dam.api.Asset, java.io.OutputStream)} instead.
     */
    void exportAsset(Node asset, OutputStream stream) throws AssetHandlerException;

    /**
     * This method exports the asset into the given <code>OutputStream</code>. The {@link Asset}s original binary is
     * served.
     *
     * @param asset  The asset to export.
     * @param stream The output stream to export into.
     * @throws AssetHandlerException If an error occurred during export.
     * @since 5.4.0
     */
    void exportAsset(Asset asset, OutputStream stream) throws AssetHandlerException;

    /**
     * This method is used by the {@link com.day.cq.dam.core.process.ExtractMetadataProcess} as part of the <i>DAM
     * Update Asset</i> workflow during import or update of an asset. Implementations must return an {@link
     * com.day.cq.dam.api.metadata.ExtractedMetadata} object, which may be empty if no metadata is extracted, or
     * contains the metadata values extracted from the binary being imported/updated at the time. The
     * <code>ExtractMetadataProcess</code> will later save the metadata contained in <code>ExtractedMetadata</code> to
     * the asset's <i>metadata</i> node (e.g. <i>/content/dam/geometrixx/banners/banner-mono.png/jcr:content/metadata</i>).
     * Implementations are free to decide which and how many metadata values are extracted.
     * <p/>
     * The method argument represents the {@link javax.jcr.Node} of type <i>nt:file</i> holding the binary content or
     * the DAM Asset node (type <i>dam:Asset</i>), for which its original rendition would be retrieved.
     *
     * @param asset the file node
     * @return all extracted metadata
     * @see com.day.cq.dam.api.Asset#getMetadata()
     * @see com.day.cq.dam.api.Asset#getMetadata(String)
     * @see com.day.cq.dam.api.Asset#getMetadataValue(String)
     * @deprecated Use {@link #extractMetadata(com.day.cq.dam.api.Asset)} instead.
     */
    ExtractedMetadata extractMetadata(Node asset);

    /**
     * This method is used by the {@link com.day.cq.dam.core.process.ExtractMetadataProcess} as part of the <i>DAM
     * Update Asset</i> workflow during import or update of an asset. Implementations must return an {@link
     * com.day.cq.dam.api.metadata.ExtractedMetadata} object, which may be empty if no metadata is extracted, or
     * contains the metadata values extracted from the binary being imported/updated at the time. The
     * <code>ExtractMetadataProcess</code> will later save the metadata contained in <code>ExtractedMetadata</code> to
     * the asset's <i>metadata</i> node (e.g. <i>/content/dam/geometrixx/banners/banner-mono.png/jcr:content/metadata</i>).
     * Implementations are free to decide which and how many metadata values are extracted.
     * <p/>
     * The method argument represents the {@link javax.jcr.Node} of type <i>nt:file</i> holding the binary content or
     * the DAM Asset node (type <i>dam:Asset</i>), for which its original rendition would be retrieved.
     *
     * @param asset The {@link Asset}, from whose original binary metadata will extracted.
     * @return The extracted metadata.
     * @since 5.4.0
     */
    ExtractedMetadata extractMetadata(Asset asset);

    /**
     * This method retrieves the graphical representation of the given file. For images the <code>BufferedImage</code>
     * of the original image is returned, for other formats the first page is retrieved as <code>BufferedImage</code>
     *
     * @param file to retrieve the image representation
     * @return <code>BufferedImage</code> if a graphical representaion exists, otherwise <code>null</code>
     * @throws java.io.IOException in case an error is thrown while fetching the buffered image
     * @since 5.3.0
     * @deprecated Use {@link #getImage(Rendition)} instead.
     */
    BufferedImage getImage(Node file) throws IOException;

    /**
     * This method retrieves the graphical representation of an {@link Asset}s given {@link Rendition}. For images the
     * <code>BufferedImage</code> of the original image is returned, for other formats the first page is retrieved as
     * <code>BufferedImage</code>
     *
     * @param rendition The rendition for which to retrieve its graphical representation.
     * @return <code>BufferedImage</code> if a graphical representation exists, otherwise <code>null</code>
     * @throws java.io.IOException in case an error is thrown while fetching the buffered image
     * @since 5.4.0
     */
    BufferedImage getImage(Rendition rendition) throws IOException;

    /**
     * This method retrieves the graphical representation of an {@link Asset}s given {@link Rendition}. For images the
     * <code>BufferedImage</code> of the original image is returned, for other formats the first page is retrieved as
     * <code>BufferedImage</code>.
     * <p>
     * If <code>maxDimension</code> is given, the handler should return an
     * image where no dimension extends the given value. This can be used to reduce
     * the memory footprint of large buffered images if the full resolution is not needed.
     *
     * @param rendition The rendition for which to retrieve its graphical representation.
     * @param maxDimension optional constraint for the maximal dimension of the image.
     * @return <code>BufferedImage</code> if a graphical representation exists, otherwise <code>null</code>
     * @throws java.io.IOException in case an error is thrown while fetching the buffered image
     * @since 5.4.0
     */
    BufferedImage getImage(Rendition rendition, Dimension maxDimension) throws IOException;

    /**
     * This method returns the mime types a particular <code>AssetHandler</code> supports.
     *
     * @return mime type, e.g. <i>image/jpg</i>
     */
    String[] getMimeTypes();

    /**
     * Handler implementations may choose to support sub asset creation for the file types it handles. The method is
     * called by the {@link com.day.cq.dam.core.process.CreateSubAssetsProcess} during import/update of an asset and its
     * binary, as part of the <i>DAM Update Asset</i> workflow. Sub assets represent fragments of the original asset,
     * for example every page of a multi-page PDF document are sub assets. Sub assets are stored as assets themselves
     * under the parent asset. Sub assets are stored in the <i>subassets</i> folder of the main asset, e.g. at
     * <i>/content/dam/geometrixx/documents/ECM Artikel.pdf/subassets</i>.
     * <p/>
     * The <code>asset</code> argument represents the {@link javax.jcr.Node} of type <i>nt:file</i> holding the binary
     * content or the DAM Asset node (type <i>dam:Asset</i>), for which its original rendition would be retrieved.
     *
     * @param asset   asset to extract sub assets
     * @param session jcr session
     * @return List constaining all paths to the newly created assets
     * @see #canHandleSubAssets()
     * @see com.day.cq.dam.api.Asset#isSubAsset()
     * @deprecated Use {@link #processSubAssets(com.day.cq.dam.api.Asset)} instead.
     */
    List<String> processSubAssets(Node asset, Session session);

    /**
     * Handler implementations may choose to support sub asset creation for the file types it handles. The method is
     * called by the {@link com.day.cq.dam.core.process.CreateSubAssetsProcess} during import/update of an asset and its
     * binary, as part of the <i>DAM Update Asset</i> workflow. Sub assets represent fragments of the original asset,
     * for example every page of a multi-page PDF document are sub assets. Sub assets are stored as assets themselves
     * under the parent asset. Sub assets are stored in the <i>subassets</i> folder of the main asset, e.g. at
     * <i>/content/dam/geometrixx/documents/ECM Artikel.pdf/subassets</i>.
     * <p/>
     * The <code>asset</code> argument represents the {@link javax.jcr.Node} of type <i>nt:file</i> holding the binary
     * content or the DAM Asset node (type <i>dam:Asset</i>), for which its original rendition would be retrieved.
     *
     * @param asset asset to extract sub assets
     * @return List constaining all paths to the newly created assets
     * @see #canHandleSubAssets()
     * @see com.day.cq.dam.api.Asset#isSubAsset()
     */
    List<String> processSubAssets(Asset asset);

    /**
     * extracts and stores the links to the assets related with provided asset.
     *
     * @param asset asset to process
     *
     * @return Iterator of {@link AssetRelation} or an empty iterator if no related assets are available
     * */
    Iterator<? extends AssetRelation> processRelated(Asset asset);
}
