/*
 * Copyright 1997-2009 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.dam.commons.handler;

import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;

import com.day.cq.dam.api.Context;
import com.day.cq.dam.api.FormatHandler;
import com.day.cq.dam.api.Processor;
import com.day.cq.dam.api.ProcessorException;
import com.day.cq.dam.commons.thumbnail.XapThumbnailsProcessor;

/**
 * Base class for format handlers.
 *
 * @author dpfister
 */
public class DefaultFormatHandler implements FormatHandler {

    /**
     * {@inheritDoc}
     *
     * The default format handler accepts all kind of data.
     */
    public boolean accepts(byte[] data, int off, int len) {
        return true;
    }

    /**
     * Return the processors responsible for detecting thumbnails. This
     * implementation detects thumbnails in &lt;xapGImg:image&gt;
     * tags.
     *
     * @return processors
     */
    protected Processor[] getThumbnailProcessors(Context context) {
        return new Processor[] { new XapThumbnailsProcessor(context) };
    }

    /**
     * Return the processors responsible for detecting metadata. This
     * implementation detects metadata in &lt;?xpacket&gt; tags.
     *
     * @return processors
     */
    protected Processor[] getMetadataProcessors(Context context) {
        return new Processor[] { new XMPProcessor(context) };
    }

    /**
     * Return all processors available for this mime type.
     *
     * @return processors
     */
    protected Processor[] getAllProcessors(Context context) {
        Processor[] t = getThumbnailProcessors(context);
        Processor[] m = getMetadataProcessors(context);

        Processor[] result = new Processor[t.length + m.length];
        System.arraycopy(t, 0, result, 0, t.length);
        System.arraycopy(m, 0, result, t.length, m.length);
        return result;
    }

    /**
     * {@inheritDoc}
     */
    public final void process(InputStream in, Context context) throws IOException {
        Processor[] processors = getAllProcessors(context);
        byte[] data = new byte[8192];
        int len;

        while ((len = in.read(data)) > 0) {
            for (Processor processor : processors) {
                processor.process(data, 0, len);
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    public BufferedImage getThumbnailImage(InputStream in)
            throws IOException, ProcessorException {

        SimpleContext context = new SimpleContext();
        Processor[] processors = getThumbnailProcessors(context);
        byte[] data = new byte[8192];
        int len;

        while ((len = in.read(data)) > 0) {
            for (Processor processor : processors) {
                processor.process(data, 0, len);
            }
        }
        BufferedImage[] thumbnails = context.getThumbnails();
        if (thumbnails != null && thumbnails.length > 0) {
            return thumbnails[0];
        }
        ProcessorException[] exceptions = context.getExceptions();
        if (exceptions != null) {
            throw exceptions[exceptions.length - 1];
        }
        return null;
    }

    /**
     * {@inheritDoc}
     */
    public InputStream getMetadata(InputStream in)
            throws IOException, ProcessorException {

        SimpleContext context = new SimpleContext();
        Processor[] processors = getMetadataProcessors(context);
        byte[] data = new byte[8192];
        int len;

        while ((len = in.read(data)) > 0) {
            for (Processor processor : processors) {
                processor.process(data, 0, len);
            }
        }
        InputStream[] metadata = context.getMetadata();
        if (metadata != null && metadata.length > 0) {
            return metadata[0];
        }
        ProcessorException[] exceptions = context.getExceptions();
        if (exceptions != null) {
            throw exceptions[exceptions.length - 1];
        }
        return null;
    }

    /**
     * Locate the pattern given in a byte array.
     *
     * @param pattern binary pattern to locate
     * @param data data to scan
     * @param off offset inside data
     * @param len number of valid bytes in data
     * @return offset where pattern was found (position following the pattern);
     *         <code>-1</code> if pattern was not found
     */
    protected int locate(byte[] pattern, byte[] data, int off, int len) {
        int i = 0;

        while (i < pattern.length && off < len) {
            if (pattern[i] == data[off]) {
                i++;
            } else {
                i = 0;
            }
            off++;
        }
        return i == pattern.length ? off : -1;
    }

    /**
     * Check whether the bytes following match the given byte array.
     *
     * @param expected byte array
     * @param actual actual bytes following
     * @return <code>true</code> if the bytes following match;
     *         <code>false</code> otherwise.
     * @throws IOException if an I/O error occurs
     */
    protected boolean matchPrefix(byte[] expected, byte[] actual) throws IOException {
        if (actual.length < expected.length) {
            return false;
        }
        for (int i = 0; i < expected.length; i++) {
            if (expected[i] != actual[i]) {
                return false;
            }
        }
        return true;
    }
}
