/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.dam.commons.process;

import com.day.cq.commons.jcr.JcrConstants;
import com.day.cq.dam.api.Asset;
import com.day.cq.dam.api.AssetManager;
import com.day.cq.dam.api.handler.AssetHandler;
import com.day.cq.dam.api.handler.store.AssetStore;
import com.day.cq.dam.commons.handler.AbstractAssetHandler;
import com.day.cq.dam.commons.util.DamUtil;
import com.day.cq.workflow.WorkflowSession;
import com.day.cq.workflow.exec.JavaProcessExt;
import com.day.cq.workflow.exec.WorkItem;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.ReferencePolicy;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.commons.mime.MimeTypeService;
import org.apache.sling.jcr.resource.JcrResourceResolverFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.UnsupportedRepositoryOperationException;
import java.util.LinkedList;
import java.util.List;

import static com.day.cq.commons.jcr.JcrConstants.JCR_CONTENT;
import static com.day.cq.commons.jcr.JcrConstants.JCR_MIMETYPE;
import static com.day.cq.commons.jcr.JcrConstants.NT_FILE;
import static com.day.cq.dam.api.DamConstants.NT_DAM_ASSET;
import static com.day.cq.dam.api.DamConstants.ORIGINAL_FILE;
import static com.day.cq.dam.api.DamConstants.RENDITIONS_FOLDER;

/**
 * The <code>AbstractAssetProcess</code> class ...
 *
 * @deprecated use {@link AbstractAssetWorkflowProcess} instead.
 */
@Component(componentAbstract = true, metatype = false)
public abstract class AbstractAssetProcess implements JavaProcessExt {

    /**
     * Logger instance for this class.
     */
    private static final Logger log = LoggerFactory.getLogger(AbstractAssetHandler.class);

    public static final String TYPE_JCR_PATH = "JCR_PATH";

    protected static final String APPLICATION_OCTET_STREAM_MIMETYPE = "application/octet-stream";

    @Reference(policy = ReferencePolicy.STATIC)
    protected MimeTypeService mimeTypeService;

    @Reference(policy = ReferencePolicy.STATIC)
    private AssetStore store;

    @Reference(policy = ReferencePolicy.STATIC)
    private JcrResourceResolverFactory jcrResolverFactory;


    /**
     * Retrieves the {@link Asset} from the payload of the given workflow <code>item</code>. If the payload type of the
     * workfow is <code>JCR_PATH</code>, the payload value is treated as the path of the asset and subsequently the
     * asset resolved and returned. If the asset is not found, <code>null</code> is returned.
     *
     * @param item    The {@link com.day.cq.workflow.exec.WorkItem} to retrieve the asset from.
     * @param session The session to access the repository with.
     * @return The resolved payload asset or <code>null</code> if not found.
     */
    protected Asset getAssetFromPayload(final WorkItem item, final Session session) {

        Asset asset = null;

        if (item.getWorkflowData().getPayloadType().equals(TYPE_JCR_PATH)) {
            final String path = item.getWorkflowData().getPayload().toString();
            final Resource resource = getResourceResolver(session).getResource(path);
            if (null != resource) {
                asset = DamUtil.resolveToAsset(resource);
            } else {
                log.error("getAssetFromPaylod: asset [{}] in payload of workflow [{}] does not exist.", path,
                        item.getWorkflow().getId());
            }
        }
        return asset;
    }

    /**
     * Retrieve the {@link Node} contained in a {@link WorkItem}'s payload, as indicated by the payload JCR path string.
     * If the node cannot be found, <code>null</code> is returned.
     *
     * @param item    The workflow item from which to get the payload.
     * @param session The JCR {@link Session} to access the repository with.
     * @return The node given by the workflow payload, or <code>null</code> if not found.
     */
    protected Node getNodeFromPayload(WorkItem item, Session session) {
        Node asset = null;
        if (item.getWorkflowData().getPayloadType().equals(TYPE_JCR_PATH)) {
            String path = item.getWorkflowData().getPayload().toString();
            try {
                if (session.itemExists(path)) {
                    asset = (Node) session.getItem(path);
                } else {
                    log.warn("getNodeFromPayload: payload node [{}] for work " +
                            "item [" + item.getId() + "] does not exist anymore", path);
                }
            } catch (RepositoryException e) {
                log.error(
                        "getNodeFromPayload: error while getting payload node [{}] " +
                                "for work item [" + item.getId() + "]: ", path, e);
            }
        }
        return asset;
    }

    protected String getMimetype(Node file) {
        String mimetype = null;
        try {
            if (file.isNodeType(NT_FILE) &&
                    file.hasNode(JCR_CONTENT) &&
                    file.hasProperty(JCR_CONTENT + "/" + JCR_MIMETYPE)) {
                mimetype = file.getProperty(JCR_CONTENT + "/" + JCR_MIMETYPE).getString();
            } else if (file.isNodeType(NT_DAM_ASSET) &&
                    file.hasProperty(JCR_CONTENT + "/" + RENDITIONS_FOLDER + "/"
                            + ORIGINAL_FILE + "/" + JCR_CONTENT + "/"
                            + JCR_MIMETYPE)) {
                mimetype = file.getProperty(JCR_CONTENT + "/" + RENDITIONS_FOLDER + "/"
                        + ORIGINAL_FILE + "/" + JCR_CONTENT + "/" + JCR_MIMETYPE)
                        .getString();
            }

            mimetype = recheck(mimetype, file);
        } catch (RepositoryException e) {
            log.error("getMimetype: error while getting mime type for file [{}]: ", safeGetPath(file), e);
        }

        return mimetype;
    }

    /**
     * This method is a temporary solution to workaround the mac webdav behaviour
     * which breaks the post processing stuff
     *
     * @param asset The {@link Node} to check.
     * @return <code>true</code> if the particular node is ready to be processed.
     * @throws RepositoryException If an error occurred accessing the repository.
     */
    protected boolean isNotReadyForProcessing(Node asset) throws RepositoryException {
        if (asset.hasProperty(JcrConstants.JCR_CONTENT + "/" + JcrConstants.JCR_DATA)) {
            Property data = asset.getProperty(JcrConstants.JCR_CONTENT + "/" + JcrConstants.JCR_DATA);
            if (data.getLength() == 0) {
                log.info("asset not ready for processing: is 0 bytes: {}", asset.getPath());
                return true;
            }
            try {
                int numRetries = 1;
                // wait some time if locked and give webdav a chance to release the lock
                while (asset.isLocked()) {
                    log.info("asset not ready for processing {}: is locked...retry {}/4", asset.getPath(), numRetries);
                    try {
                        Thread.sleep(500);
                    } catch (InterruptedException e) {
                        // ignore
                    }
                    if (++numRetries > 4) {
                        log.info("asset not ready for processing: is locked: {}", asset.getPath());
                        return true;
                    }
                }
            } catch (UnsupportedRepositoryOperationException e) {
                log.info(
                        "isNotReadyForProcessing: repository does not support locking; asset: [{}]: ",
                        safeGetPath(asset),
                        e
                );
            }
            return false;
        } else {
            return true;
        }
    }

    /**
     * This method allows you to get by passing a argument key the matching values. E.g.:<br/> args[0] =
     * dimension:200:200 args[1] = dimension: 150:150 args[2] = name:cq5dam.web <br/> by calling <code>dimension</code>
     * the values form arg 1 and 2 are returned
     *
     * @param key       argument key
     * @param arguments arguments
     * @return list of values
     */
    protected List<String> getValuesFromArgs(String key, String arguments[]) {
        final List<String> values = new LinkedList<String>();
        for (String str : arguments) {
            if (str.startsWith(key + ":")) {
                final String mt = str.substring((key + ":").length()).trim();
                values.add(mt);
            }
        }
        return values;
    }


    /**
     * It might happen that the webdav servlet does not detect the mimetype properly. So rechecking a lowwercased file
     * name makes sense.
     *
     * @param mimeType
     * @param file
     * @return
     * @throws RepositoryException
     */
    protected String recheck(String mimeType, Node file) throws RepositoryException {
        String mType = mimeType;
        if (mimeType == null || mimeType.equals(APPLICATION_OCTET_STREAM_MIMETYPE)) {
            String name = file.getName().toLowerCase();
            if (mimeTypeService.getMimeType(name) != null) {
                mType = mimeTypeService.getMimeType(name);
            }
        }
        return mType;
    }

    /**
     * Return the path of an asset node or the literal <code>(unknown)</code> if getting the path fails.
     *
     * @param node node
     * @return path or <code>null</code>
     */
    protected String safeGetPath(Node node) {
        try {
            return node.getPath();
        } catch (RepositoryException e) {
            log.warn("safeGetPath: error while getting path from node: ", e);
            return "(unknown)";
        }
    }

    protected ResourceResolver getResourceResolver(final Session session) {
        return jcrResolverFactory.getResourceResolver(session);
    }

    protected AssetManager getAssetManager(final Session session) {
        return getResourceResolver(session).adaptTo(AssetManager.class);
    }

    protected AssetHandler getAssetHandler(final String mimeType) {
        return store.getAssetHandler(mimeType);
    }

    public void execute(WorkItem workItem, WorkflowSession workflowSession)
            throws Exception {
        // noop
    }
}
