/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2015 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.day.cq.dam.commons.ui.editor.metadata;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.jcr.RepositoryException;

import org.apache.commons.lang.ArrayUtils;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Service;
import org.apache.sling.api.SlingException;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ValueMap;

import com.day.cq.dam.api.Asset;
import com.day.cq.dam.api.DamConstants;
import com.day.cq.dam.commons.util.SchemaFormHelper;
import com.day.cq.workflow.collection.ResourceCollection;
import com.day.cq.dam.api.ui.editor.metadata.MetadataEditorHelper;

/**
 * @see {@link MetadataEditorHelper}.
 */
@Service
@Component
public class MetadataEditorHelperImpl implements MetadataEditorHelper {

    private static final String MIMETYPE_MAPPING_PATH = "/etc/dam/metadataeditor/mimetypemappings";

    private static final String FOLDER_SPECIFIC_METADATA_FORM = "jcr:content/metadataSchema";

    private static final String REQUIRED = "required";

    private static final String METADATA_PROPERTY_NAME = "name";

    private static final String COLLECTION_FORM = "collection";

    private static final String DEFAULT_FORM = "default";

    public Resource getEditorFormResource(Resource... resources) {
        return getFormResource(resources);
    }

    public List<Resource> getInvalidFormItems(Resource resource) {
        return getEmptyMandatoryMetadata(resource);
    }

    private List<Resource> getEmptyMandatoryMetadata(Resource resource) {
        Resource formResource = getFormResource(resource);
        List<Resource> items = new ArrayList<Resource>();
        List<Resource> emptyMandatoryFields = new ArrayList<Resource>();
        traverseForm(formResource, items);
        for (Resource each : items) {
            ValueMap vm = each.adaptTo(ValueMap.class);
            boolean required = vm.get(REQUIRED, false);
            String name = vm.get(METADATA_PROPERTY_NAME, String.class);
            if (null != name && !"".equals(name) && required) {
                if (name.startsWith("./")) {
                    name = name.substring(2);
                }
                ValueMap vmLoad = resource.adaptTo(ValueMap.class);
                if (null == vmLoad.get(name)) {
                    emptyMandatoryFields.add(each);
                }
            }
        }
        return emptyMandatoryFields;
    }

    private void traverseForm(Resource resource, List<Resource> items) {
        if (null != resource) {
            ValueMap vm = resource.adaptTo(ValueMap.class);
            if (vm.containsKey(METADATA_PROPERTY_NAME) && vm.containsKey(REQUIRED)) {
                items.add(resource);
            }
            if (resource.hasChildren()) {
                Iterator<Resource> children = resource.listChildren();
                while (children.hasNext()) {
                    Resource child = children.next();
                    traverseForm(child, items);
                }
            }
        }
    }

    private Asset[] convertToAssets(Resource... resources) throws IllegalArgumentException {
        Asset[] assets = new Asset[resources.length];
        for (int i = 0; i < resources.length; i++) {
            Asset each = resources[i].adaptTo(Asset.class);
            if (null != each) {
                assets[i] = each;
            } else {
                throw new IllegalArgumentException("All resources are not adaptable to asset");
            }
        }
        return assets;
    }

    private String getMimeType(Asset... assets) {

        if (null == assets || assets.length == 0) {
            throw new IllegalArgumentException("Assets must not be null or empty");
        }

        ResourceResolver resourceResolver = assets[0].adaptTo(Resource.class).getResourceResolver();

        MediaType commonType = null;
        for (Asset each : assets) {
            String eachType = each.getMimeType();
            eachType = (eachType==null) ? "application/octet-stream" : eachType;
            String exposedType = getExposedmimetype(eachType, resourceResolver);
            if (null != commonType) {
                commonType = commonType.getCommonType(new MediaType(exposedType));
            } else {
                commonType = new MediaType(exposedType);
            }
        }
        return commonType.getMimeType();

    }

    private String getExposedmimetype(String mimetype, ResourceResolver resourceResolver) {

        final String MIMETYPES = "mimetypes";
        final String EXPOSED_MIMETYPE = "exposedmimetype";

        Resource mimeTypeMappingsRes = resourceResolver.getResource(MIMETYPE_MAPPING_PATH);
        if (mimeTypeMappingsRes != null) {
            Iterator<Resource> children = mimeTypeMappingsRes.listChildren();
            while (children.hasNext()) {
                Resource child = children.next();
                ValueMap vm = child.adaptTo(ValueMap.class);
                if (vm.containsKey(MIMETYPES)) {
                    String mimeTypes[] = vm.get(MIMETYPES, String[].class);
                    if (ArrayUtils.contains(mimeTypes, mimetype)) {
                        mimetype = vm.get(EXPOSED_MIMETYPE, String.class);
                        return mimetype;
                    }
                }
            }
        }
        return mimetype;
    }

    private static class MediaType {
        private final String topLevelType;

        private final String subType;

        private final String mimeType;

        MediaType(final String mediaType) {
            if (mediaType == null || mediaType.length() == 0) {
                throw new IllegalArgumentException("mediaType must not be null or empty.");
            }
            String[] parts = mediaType.split("/");
            if (parts.length > 2) {
                throw new IllegalArgumentException("Invalid mediaType.");
            }
            this.mimeType = mediaType;
            this.topLevelType = parts[0];
            this.subType = (parts.length > 1 && parts[1].length() > 0) ? parts[1] : null;
        }

        private MediaType(final String topLevelType, final String subType) {
            this.topLevelType = topLevelType;
            this.subType = subType;
            String mimeType = "";
            if (null != this.topLevelType) {
                mimeType = mimeType + this.topLevelType;
                if (null != this.subType) {
                    mimeType = mimeType + '/' + this.subType;
                }
            }
            this.mimeType = mimeType;
        }

        MediaType getCommonType(final MediaType other) {
            if (this.topLevelType != null && this.topLevelType.equals(other.topLevelType)) {
                if (this.subType != null && this.subType.equals(other.subType)) {
                    return this;
                } else {
                    return new MediaType(this.topLevelType, null);
                }
            } else {
                return new MediaType(null, null);
            }
        }

        String getMimeType() {
            return this.mimeType;
        }
    }

    String getFolderSpecificMetadataForm(Resource resource, String mimeType) {
        Resource parent = resource.getParent();
        while (parent != null && parent.getChild(FOLDER_SPECIFIC_METADATA_FORM) == null && (!parent.getPath().equals(DamConstants.MOUNTPOINT_ASSETS))) {
            parent = parent.getParent();
        }
        if(parent == null) {
            return null;
        }
        ValueMap vm = parent.adaptTo(ValueMap.class);
        String folderMetadata = vm.get(FOLDER_SPECIFIC_METADATA_FORM, String.class);
        String assetSpecificForm = null;
        if (null != folderMetadata) {
            assetSpecificForm = folderMetadata + "/" + mimeType;
        } else {
            return null;
        }
        ResourceResolver resolver = resource.getResourceResolver();
        while ( resolver.resolve(assetSpecificForm).getResourceType().equals(Resource.RESOURCE_TYPE_NON_EXISTING) && assetSpecificForm !=null ) {
            assetSpecificForm = assetSpecificForm.substring(0,assetSpecificForm.lastIndexOf("/"));
        }
        return assetSpecificForm;
    }



    private Resource getAbsoluteFormResource(ResourceResolver resolver, String relativeFormPath) {
        String[] baseFormPaths = SchemaFormHelper.getBaseFormPaths(resolver);
        relativeFormPath = getRelativeFormPath(relativeFormPath, baseFormPaths);
        do{
            for(String baseFormPath : baseFormPaths){
                String path = baseFormPath + relativeFormPath;
                Resource res = resolver.getResource(path);
                if(res == null){
                    // checking for lower case for backward compatibility
                    res = resolver.getResource(path.toLowerCase());
                }
                if(res != null && ( res.getChild("items/tabs") != null || relativeFormPath.trim().isEmpty())){
                    return res;
                }
            }
            relativeFormPath = relativeFormPath.trim().isEmpty() ? "" : relativeFormPath.substring(0, relativeFormPath.lastIndexOf('/'));
        } while(!relativeFormPath.trim().isEmpty());

        return null;
    }


    private final Resource mergeResources(Resource resource) {
        // Merge forms
        List<Resource> masterTabList;
        try {
            masterTabList = SchemaFormHelper.getMasterForms(resource);
        } catch (RepositoryException e) {
            throw new SlingException("unable to get master forms ", e);
        }
        Resource rootTabRes = null;
        if (!masterTabList.isEmpty()) {
            Resource root = masterTabList.get(0);
            rootTabRes = root.getChild("items/tabs");
            for (int i = 1; i < masterTabList.size(); i++) {
                Resource formRes = masterTabList.get(i);
                Resource formTabRes = formRes.getChild("items/tabs");
                rootTabRes = SchemaFormHelper.mergeFormTabResource(rootTabRes, formTabRes);
            }
        }

        // get form for current Resource
        Resource currentFormResource = resource;
        Resource currentTabListRes = currentFormResource.getChild("items/tabs");
        if (rootTabRes != null) {
            rootTabRes = SchemaFormHelper.mergeFormTabResource(rootTabRes, currentTabListRes);
        } else {
            rootTabRes = currentTabListRes;
        }

        return rootTabRes;
    }

    private Resource getFormResource(Resource... resources) {
        if (null == resources || resources.length == 0) {
            throw new IllegalArgumentException("Failure: No resource(s) passed.");
        }

        String relativeFormPath = "";
        ResourceResolver resolver = resources[0].getResourceResolver();

        String mimeType = getMimeType(convertToAssets(resources));

        if (resources.length == 1 && resources[0].adaptTo(ResourceCollection.class) != null) {
            relativeFormPath = "/" + COLLECTION_FORM;
        } else {
            if (resources.length == 1) {
                relativeFormPath = getFolderSpecificMetadataForm(resources[0], getMimeType(convertToAssets(resources)));
            }
            if (null == relativeFormPath) {
                relativeFormPath = "/" + DEFAULT_FORM + "/" + mimeType;
            }
        }
        Resource res = getAbsoluteFormResource(resolver, relativeFormPath);

        return mergeResources(res);
    }

    private static String getRelativeFormPath(String formPath, String[] baseFormPaths){
        for(String baseFormPath : baseFormPaths){
            if (formPath.startsWith(baseFormPath)){
                String relativePath = formPath.substring(baseFormPath.length());
                relativePath = relativePath.startsWith("/") ? relativePath : relativePath + "/";
                relativePath = relativePath.endsWith("/") ? relativePath.substring(relativePath.length() -1 ) : relativePath;
                return relativePath;
            }
        }
        return formPath;
    }

}
