/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.dam.commons.util;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.net.URISyntaxException;
import java.net.URI;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.PropertyType;
import javax.jcr.RepositoryException;
import javax.jcr.Value;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.cq.dam.api.Asset;
import com.day.cq.dam.commons.util.impl.AssetReferenceProvider;
import com.day.cq.wcm.api.reference.Reference;

/**
 * The <code>AssetReferenceSearch</code> searches in pages for referenced assets.
 */
public class AssetReferenceSearch {

    /**
     * The logging facility.
     */
    private static final Logger log = LoggerFactory.getLogger(AssetReferenceSearch.class);

    private final Node node;

    private final String searchPath;

    private final ResourceResolver resolver;

    /**
     * The constructor.
     *
     * @param node node to start search for references
     * @param searchPath search for assets starting with <code>searchPath</code>
     * @param resolver resource resolver
     */
    public AssetReferenceSearch(Node node, String searchPath, ResourceResolver resolver) {
        this.node = node;
        this.searchPath = searchPath;
        this.resolver = resolver;
    }

    /**
     * Search all asset references
     *
     * @return map containing all asset refs
     */
    public Map<String, Asset> search() {
        Map<String, Asset> assetRefs =  new HashMap<String, Asset>();
        Pattern pattern = getPattern(searchPath);
        search(node, assetRefs, pattern);
        return assetRefs;
    }

    protected void search(Node node, Map<String, Asset> assetRefs, Pattern pattern) {
        try {
            for (PropertyIterator pIter = node.getProperties(); pIter.hasNext();) {
                Property p = pIter.nextProperty();
                // only check string and name properties
                if (p.getType() == PropertyType.STRING || p.getType() == PropertyType.NAME) {
                    boolean decode = p.getType() == PropertyType.STRING;
                    if (p.getDefinition().isMultiple()) {
                        for (Value v : p.getValues()) {
                            String value = v.getString();
                            if (pattern.matcher(value).find()) {
                                if (decode) {
                                    value = tryDecode(value);
                                }
                                Set<String> refs = new HashSet<String>();
                                if (value.startsWith("/")) {
                                    // looks like just a single path
                                    refs.add(value);
                                } else {
                                    // ref might be somewhere in the string
                                    getRefs(value, refs, decode);
                                }
                                for (String ref : refs) {
                                     if (resolver.getResource(ref) != null &&
                                        resolver.getResource(ref).adaptTo(Asset.class) != null) {
                                        assetRefs.put(ref, resolver.getResource(ref).adaptTo(Asset.class));
                                     }
                                }
                            }
                        }
                    } else {

                        String value = p.getString();

                        //check to see if the image is dynamic media "IS" if so remove image serving context
                        String isImageContext = "/is/image";
                        if (value.startsWith(isImageContext)) {
                            value = value.split(isImageContext)[1];
                        }
                    	
                    	/*
                		 * Handle references for s7 set types
                		 */
                        Set<String> refs = new HashSet<String>();
                        if (pattern.matcher(value).find()) {
                            Resource r = resolver.getResource(value);

                            if (r != null && (S7SetHelper.isS7Set(r) || S7SetHelper.isS7Video(r))) {
                                AssetReferenceProvider refProvider = new AssetReferenceProvider();
                                List<Reference> s7SetRefs = refProvider.findReferences(r);

                                int i = 0;
                                int len = s7SetRefs.size();
                                String path = value;

                                //add set
                                refs.add(decode ? tryDecode(path) : path);

                                //add set members
                                for (i = 0; i < len; i++) {
                                    path = s7SetRefs.get(i).getResource().getPath();
                                    refs.add(decode ? tryDecode(path) : path);
                                }
                            }

                            Matcher matcher = pattern.matcher(value);
                            if (matcher.find()) {
                                //Set<String> refs = new HashSet<String>();

                                if (value.startsWith("/")) {
                                    // looks like just a single path
                                    refs.add(decode ? tryDecode(value) : value);
                                } else {
                                    // ref might be somewhere in the string
                                    getRefs(value, refs, decode);
                                }
                                for (String ref : refs) {
                                    if (resolver.getResource(ref) != null &&
                                            resolver.getResource(ref).adaptTo(Asset.class) != null) {
                                        assetRefs.put(ref, resolver.getResource(ref).adaptTo(Asset.class));
                                    }
                                }
                            }
                        }
                    }
                }
            }
        } catch (RepositoryException re) {
            log.warn("Error occured while reading properties");
        }

        try {
            for (NodeIterator nItr = node.getNodes(); nItr.hasNext();) {
                Node n = nItr.nextNode();
                search(n, assetRefs, pattern);
            }
        } catch (RepositoryException re) {
            log.warn("Error occured while reading nodes");
        }
    }

    private String tryDecode(String url) {
        try {
            return new URI(url).getPath();
        } catch(URISyntaxException e) {
            return url;
        }
    }

    /**
     * Search for asset paths in text
     *
     * @param value text as string
     * @param refs set to which found asset paths are added
     */
    private void getRefs(String value, Set<String> refs, boolean decode) {
        int startPos = value.indexOf(searchPath, 1);
        while (startPos != -1) {
            char charBeforeStartPos = value.charAt(startPos - 1);
            if (charBeforeStartPos == '\'' || charBeforeStartPos == '"') {
                int endPos = value.indexOf(charBeforeStartPos, startPos);
                if (endPos > startPos) {
                    String ref = value.substring(startPos, endPos);
                    refs.add(decode? tryDecode(ref) : ref);
                    startPos = endPos;
                }
            }
            startPos = value.indexOf(searchPath, startPos + 1);
        }
    }

    /**
     * Returns the replacement pattern for the rewrite method.
     *
     * @param path source path
     * @return replacement pattern
     */
    protected Pattern getPattern(String path) {
            return Pattern.compile("(.[\"']|^|^[\"'])(" + path + ")\\b");
    }
}
