package com.day.cq.dam.commons.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import com.adobe.granite.asset.core.impl.AssetConstants;
import com.day.cq.dam.api.Rendition;
import com.day.cq.wcm.api.WCMException;
import org.apache.sling.api.resource.PersistenceException;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceResolverFactory;
import org.apache.sling.jcr.resource.JcrResourceConstants;
import org.apache.sling.jcr.resource.JcrResourceUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.cq.commons.Language;
import com.day.cq.commons.LanguageUtil;
import com.day.cq.commons.jcr.JcrConstants;
import com.day.cq.dam.api.Asset;
import com.day.cq.dam.api.DamConstants;
import com.day.cq.wcm.api.PageManager;
import com.day.cq.wcm.api.PageManagerFactory;
import com.day.text.Text;

import static com.day.cq.commons.jcr.JcrConstants.JCR_CONTENT;

/**
 * This class provides utility method for Language Copy used by sites
 * 
 * @author batla
 */
public class DamLanguageUtil {

    private static final Logger log = LoggerFactory.getLogger(DamLanguageUtil.class);

    private static final String CONTENT_FRAGMENT = "contentFragment";
    private static final String ATTRIBUTE_DESTINATION_LANGUAGE_COPY_PATH = "dam:destinationLanguageCopy";
    public static final String ATTRIBUTE_ASSET_UPDATE_REQUIRED = "dam:assetUpdateRequired";
    private static final String ATTRIBUTE_SMART_ASSET_UPDATE_SOURCE = "dam:smartAssetUpdateSource";
    private static final String ATTRIBUTE_EXTRACT_METADATA = "dam:extractMetadata";
    private static final String ASSET_VERSION_MESSAGE = "Created by Asset Update Translation";
    private static final String ASSET_PERFORMANCE_NODE_RELATIVE_PATH = "/" + JCR_CONTENT + "/" + "performance";
    private static final String ASSET_USAGE_NODE_RELATIVE_PATH = "/" + JCR_CONTENT + "/" + DamConstants.USAGES_FOLDER;


    /***
     * This method returns true if language copy of an asset exists, for the
     * asked locale
     * 
     * @param assetPath The path of an asset for which language copy is asked
     * @param languageCode Language for which language copy is asked
     * @param resolver ResourceResolver
     * @return
     */
    public static boolean hasLanguageCopy(final String assetPath,
            final String languageCode, final ResourceResolver resolver) {

        Asset asset = findLanguageCopy(assetPath, languageCode, resolver);
        if (asset != null) return true;
        return false;
    }
    /**
     * This method returns the Language copy asset if language copy exists, for
     * the asked locale
     * 
     * @param assetPath The path of an asset for which language copy is asked
     * @param languageCode Language for which language copy is asked
     * @param resolver ResourceResolver
     * @return
     */
    public static Asset getLanguageCopy(final String assetPath,
            final String languageCode, final ResourceResolver resolver) {

        return findLanguageCopy(assetPath, languageCode, resolver);

    }

    /***
     * This method creates language copy of an asset/folder
     *
     * @param resourceResolver
     * @param pageManagerFactory
     * @param sourcePath - source for creating language copy
     * @param targetLanguageCodes - array of language codes
     * @return
     */
    public static List<String> createLanguageCopy(final ResourceResolver resourceResolver,
        final PageManagerFactory pageManagerFactory, final String sourcePath, final String[] targetLanguageCodes) {

        Session session = resourceResolver.adaptTo(Session.class);
        PageManager pageManager = pageManagerFactory.getPageManager(resourceResolver);
        Resource sourceResource = resourceResolver.getResource(sourcePath);
        List<String> createdCopies = new ArrayList<String>();
        String languageCopyPath = "";
        String contentPath = "";

        if (targetLanguageCodes == null || targetLanguageCodes.length == 0 || targetLanguageCodes[0].trim().length() == 0) {
            log.error("Failed to load destination language from payload.");
            return null;
        }

        String root = LanguageUtil.getLanguageRoot(sourcePath);
        String parentOfRoot = null;
        boolean createNewLanguageRoot = false;
        if (root == null) {
            log.debug("Language root does not exist for asset at path: {} and would be created. ", sourcePath);

            //CQ-61450 User creates a language copy of a page from sites and language root for the asset (referenced in site) does not exist
            if (Text.getRelativeParent(sourcePath, 1).equals(DamConstants.MOUNTPOINT_ASSETS)) {
                //Asset is directly under DamConstants.MOUNTPOINT_ASSETS
                createNewLanguageRoot = true;
                parentOfRoot = DamConstants.MOUNTPOINT_ASSETS;
                root = DamConstants.MOUNTPOINT_ASSETS;
            } else if (sourcePath.startsWith(DamConstants.MOUNTPOINT_ASSETS + "/")) {
                //Asset follows structure DamConstants.MOUNTPOINT_ASSETS/<website>/
                int parentOfRootPathLength = sourcePath.indexOf('/', DamConstants.MOUNTPOINT_ASSETS.length() + 1);
                int oldRootPathLength = sourcePath.indexOf('/', parentOfRootPathLength);
                if (parentOfRootPathLength < 0 || sourcePath.length() <= parentOfRootPathLength) {
                    return createdCopies;
                }
                createNewLanguageRoot = true;
                parentOfRoot = sourcePath.substring(0, parentOfRootPathLength);

                if (oldRootPathLength > 0 && sourcePath.length() > oldRootPathLength) {
                    root = sourcePath.substring(0, oldRootPathLength);
                }
            }

            log.info("Parent of New Language root at path {} added for asset at path {}", parentOfRoot, sourcePath);
            contentPath = sourcePath.replaceFirst(parentOfRoot, "");
        } else {
            contentPath = sourcePath.replaceFirst(root, "");
            parentOfRoot = Text.getRelativeParent(root, 1);
        }

        for (int i = 0; i < targetLanguageCodes.length; i++) {
            String targetPath = "";
            String languageRootPath = "";

            String strDestinationLanguage = getDestinationLanguageWithAllowedDelimiters(parentOfRoot,
                targetLanguageCodes[i], resourceResolver);

            languageRootPath = parentOfRoot + "/" + strDestinationLanguage;
            targetPath = languageRootPath + contentPath;

            try {
                if (contentPath.trim().length() > 0) {
                    String pathToCreate = Text.getRelativeParent(
                            targetPath, 1);
                    String nodeType = "sling:Folder";

                    if (session.getNode(root).isNodeType(
                            "sling:OrderedFolder")) {
                        nodeType = "sling:OrderedFolder";
                    }
                    JcrResourceUtil.createPath(pathToCreate, nodeType,
                            nodeType, session, false);
                }
                if (!session.nodeExists(targetPath)) {
                    Resource destinationResource = pageManager.copy(sourceResource, targetPath, null, false,
                        false, false);
                    deleteInsightData(destinationResource.getPath(), resourceResolver);
                    //changing the title of the folder
                    Node targetNode = session.getNode(languageRootPath);
                    if (targetNode != null) {
                        if (!targetNode.hasNode(JcrConstants.JCR_CONTENT)) {
                            targetNode.addNode(JcrConstants.JCR_CONTENT, JcrConstants.NT_UNSTRUCTURED);
                        }
                        Node content = targetNode.getNode(JcrConstants.JCR_CONTENT);

                        if (!content.hasProperty(JcrConstants.JCR_TITLE)) {
                            String displayLanguage = LanguageUtil.getLanguage(strDestinationLanguage).
                                    getLocale().getDisplayLanguage();
                            content.setProperty(JcrConstants.JCR_TITLE, displayLanguage);
                        }

                        //set JCR_TITLE for all folders in the created contentPath and language root
                        String tempAddTitlePath = contentPath.substring(0, contentPath.lastIndexOf('/'));
                        while (tempAddTitlePath.length() > 0) {
                            Node tempNode = session.getNode(languageRootPath + tempAddTitlePath);
                            if (tempNode != null) {
                                if (!tempNode.hasNode(JcrConstants.JCR_CONTENT)) {
                                    tempNode.addNode(JcrConstants.JCR_CONTENT, JcrConstants.NT_UNSTRUCTURED);
                                }
                                Node tempNodeContent = tempNode.getNode(JcrConstants.JCR_CONTENT);
                                String sourceNodeForTitle = null;
                                if (createNewLanguageRoot) {
                                    sourceNodeForTitle = parentOfRoot + tempAddTitlePath;
                                } else {
                                    sourceNodeForTitle = root + tempAddTitlePath;
                                }
                                if (!tempNodeContent.hasProperty(JcrConstants.JCR_TITLE)) {
                                    String tempNodeTitle = UIHelper.getTitle(resourceResolver.
                                            getResource(sourceNodeForTitle));
                                    tempNodeContent.setProperty(JcrConstants.JCR_TITLE, tempNodeTitle);
                                }
                                tempAddTitlePath = tempAddTitlePath.substring(0, tempAddTitlePath.lastIndexOf('/'));
                            }
                        }
                    }
                } else {
                    log.info(
                        "Could not create language copy for assets at path: " + targetPath +
                            ", resource already exists. Updating language copy.");
                    Asset sourceAsset = resourceResolver.getResource(sourcePath).adaptTo(Asset.class);
                    Asset targetAsset = resourceResolver.getResource(targetPath).adaptTo(Asset.class);
                    if (sourceAsset != null && targetAsset != null) {
                        if (DamLanguageUtil.isSmartAssetUpdateRequired(sourceAsset, targetAsset)) {
                            DamLanguageUtil.addSmartAssetUpdateSource(targetAsset, sourcePath);
                        }
                    }
                }
                createdCopies.add(targetPath);
            } catch (Exception e) {
                log.error(
                    "error while creating language copy for assets at path: "
                                + languageCopyPath + "{}", e);
            }
        }
        return createdCopies;

    }

    private static Asset findLanguageCopy(final String assetPath,
            final String languageCode, final ResourceResolver resolver) {

        Resource assetResource = resolver.getResource(assetPath);
        if (assetResource == null || assetResource.adaptTo(Asset.class) == null) {
            return null;
        } 
        
        Asset asset = null;        
        String languageRootPath = LanguageUtil.getLanguageRoot(assetPath);

        if(languageRootPath == null){
            return null;
        }

        String contentPath = assetPath.replaceFirst(languageRootPath, "");
        String languageRootParentPath = Text.getRelativeParent(
                languageRootPath, 1);
        String assetPathLC = languageRootParentPath + "/" + languageCode + contentPath;
        Resource assetResourceLC = resolver.getResource(assetPathLC);
        if (assetResourceLC != null) {
            asset = assetResourceLC.adaptTo(Asset.class);
        }
        return asset;
    }
    
    /**
     * Returns the language root for the given asset path by only analyzing the
     * path names starting at the root. For this implementation, language root
     * for dam was suffix of folder name joined by "-" ex. geometrixx-en.
     * 
     * @param path path
     * @return the language root or <code>null</code> if not found
     *
     * @deprecated since 6.2, use {@link com.day.cq.commons.LanguageUtil} instead
     */
    @Deprecated
    public static String getLanguageRoot(String path) {
        if (path == null || path.length() == 0 || path.equals("/")) {
            return null;
        }
        path = path + "/";
        int idx = path.indexOf('/', 1);
        while (idx > 0) {
            String subPath = path.substring(0, idx);
            String name = Text.getName(subPath);
            if (name.contains("-")) {
                String namelocale = name.substring(name.lastIndexOf("-") + 1);
                Locale locale = LanguageUtil.getLocale(namelocale);
                if (locale != null) {
                    return subPath;
                }
            }
            idx = path.indexOf('/', idx + 1);
        }
        return null;
    }
    
    /**
     * Returns the language for the given asset path by only analyzing the
     * path names starting at the root. For this implementation, language
     * root for dam was suffix of folder name joined by "-" ex. geometrixx-en.
     * 
     * @param path path
     * @return the language or <code>null</code> if not found
     *
     * @deprecated since 6.2, use {@link com.day.cq.commons.LanguageUtil} instead
     */
    @Deprecated
    public static Language getLanguage(String path) {
        path = getLanguageRoot(path);
        if (path != null) {
            String languageRoot = path.substring(path.lastIndexOf("-") + 1);
            return new Language(languageRoot);
        }
        return null;
    }

    /**
     * Modified version of com.day.cq.wcm.core.impl.LanguageManagerImpl for Resources
     *
     * Returns a collection of language root pages for the given page. Note that
     * only the path names are respected for the determination of the language.
     *
     * @param resolver resource resolver
     * @param path path of the current page
     * @return collection of language root paths
     */
    public static Collection<Resource> getLanguageRoots(ResourceResolver resolver, String path) {
        String root = LanguageUtil.getLanguageRoot(path);
        if (root == null) {
            return Collections.emptySet();
        }
        String parent = Text.getRelativeParent(root, 1);
        Resource parentResource = resolver.getResource(parent);
        if (parentResource == null) {
            return Collections.emptySet();
        }
        List<Resource> resources = new LinkedList<Resource>();
        Iterator<Resource> iter = resolver.listChildren(parentResource);
        while (iter.hasNext()) {
            Resource r = iter.next();
            Locale locale = LanguageUtil.getLocale(Text.getName(r.getPath()));
            if (locale != null) {
                    resources.add(r);
            }
        }
        return resources;
    }

    /***
     * This method creates update language copy of an asset/folder
     *
     * @param resourceResolver
     * @param pageManagerFactory
     * @param sourcePath          - source for creating language copy
     * @param targetLanguageCode - destination language code
     * @param prefixPath - Root path where language copies are created
     * @return
     */
    public static String createUpdateLanguageCopy(final ResourceResolver resourceResolver, final PageManagerFactory pageManagerFactory, final String sourcePath,
                                                  final String targetLanguageCode, String prefixPath) {

        Session session = resourceResolver.adaptTo(Session.class);
        PageManager pageManager = pageManagerFactory.getPageManager(resourceResolver);
        Resource sourceResource = resourceResolver.getResource(sourcePath);
        String createdCopy = "";
        String languageCopyPath = "";
        String contentPath = "";

        if (targetLanguageCode == null || targetLanguageCode.trim().length() == 0) {
            log.error("Failed to load destination language from payload.");
            return null;
        }

        String root = LanguageUtil.getLanguageRoot(sourcePath);
        String parentOfRoot = null;
        boolean createNewLanguageRoot = false;
        if (root == null) {
            log.debug("Language root does not exist for asset at path: {} and would be created. ", sourcePath);

            //CQ-61450 User creates a language copy of a page from sites and language root for the asset (referenced in site) does not exist
            if (Text.getRelativeParent(sourcePath, 1).equals(DamConstants.MOUNTPOINT_ASSETS)) {
                //Asset is directly under DamConstants.MOUNTPOINT_ASSETS
                createNewLanguageRoot = true;
                parentOfRoot = DamConstants.MOUNTPOINT_ASSETS;
                root = DamConstants.MOUNTPOINT_ASSETS;
            } else if (sourcePath.startsWith(DamConstants.MOUNTPOINT_ASSETS + "/")) {
                //Asset follows structure DamConstants.MOUNTPOINT_ASSETS/<website>/
                int parentOfRootPathLength = sourcePath.indexOf('/', DamConstants.MOUNTPOINT_ASSETS.length() + 1);
                int oldRootPathLength = sourcePath.indexOf('/', parentOfRootPathLength);
                if (parentOfRootPathLength < 0 || sourcePath.length() <= parentOfRootPathLength) {
                    return createdCopy;
                }
                createNewLanguageRoot = true;
                parentOfRoot = sourcePath.substring(0, parentOfRootPathLength);

                if (oldRootPathLength > 0 && sourcePath.length() > oldRootPathLength) {
                    root = sourcePath.substring(0, oldRootPathLength);
                }
            }

            log.info("Parent of New Language root at path {} added for asset at path {}", parentOfRoot, sourcePath);
            contentPath = sourcePath.replaceFirst(parentOfRoot, "");
        } else {
            contentPath = sourcePath.replaceFirst(root, "");
            parentOfRoot = Text.getRelativeParent(root, 1);
        }

        String targetPath = "";
        String languageRootPath = "";

        String strDestinationLanguage = getDestinationLanguageWithAllowedDelimiters(prefixPath + parentOfRoot,
            targetLanguageCode, resourceResolver);

        languageRootPath = prefixPath + parentOfRoot + "/" + strDestinationLanguage;
        targetPath = languageRootPath + contentPath;

        try {
//            if (contentPath.trim().length() > 0) {
                String pathToCreate = Text.getRelativeParent(
                        targetPath, 1);
                String nodeType = "sling:Folder";

                if (session.getNode(root).isNodeType(
                        "sling:OrderedFolder")) {
                    nodeType = "sling:OrderedFolder";
                }
                JcrResourceUtil.createPath(pathToCreate, nodeType,
                        nodeType, session, false);
//            }
            if (DamUtil.isAsset(sourceResource)) {
                Resource destinationResource = pageManager.copy(sourceResource, targetPath, null, false, true, false);
                deleteInsightData(destinationResource.getPath(), resourceResolver);
                removeAllRenditionsInsideResource(destinationResource);
                String destinationForTemporaryAsset = parentOfRoot + "/" + strDestinationLanguage + contentPath;
                setDestinationLanguageCopyPath(destinationResource, destinationForTemporaryAsset);
                createdCopy = destinationResource.getPath();
            } else if (isFolder(sourceResource)) {
                //Add destination Language copy path to all assets inside folder
                String sourceFolderPath = sourceResource.getPath();
                String destinationFolderPath = parentOfRoot + "/" + targetLanguageCode + contentPath;
                Iterator<Asset> assetIterator = DamUtil.getAssets(sourceResource);
                while (assetIterator.hasNext()) {
                    String assetPath = assetIterator.next().getPath();
                    String destinationAssetPath = assetPath.replaceFirst(sourceFolderPath, destinationFolderPath);
                    String temporaryAssetPath = prefixPath + destinationAssetPath;

                    Resource assetResource = resourceResolver.getResource(assetPath);
                    Resource destinationResource = pageManager.copy(assetResource, temporaryAssetPath, null, false, true, false);
                    deleteInsightData(destinationResource.getPath(), resourceResolver);
                    removeAllRenditionsInsideResource(destinationResource);
                    setDestinationLanguageCopyPath(destinationResource, destinationAssetPath);
                }
                createdCopy = targetPath;
            }
            //changing the title of the folder
            Node targetNode = session.getNode(languageRootPath);
            if (targetNode != null) {
                if (!targetNode.hasNode(JcrConstants.JCR_CONTENT)) {
                    targetNode.addNode(JcrConstants.JCR_CONTENT, JcrConstants.NT_UNSTRUCTURED);
                }
                Node content = targetNode.getNode(JcrConstants.JCR_CONTENT);

                    if (!content.hasProperty(JcrConstants.JCR_TITLE)) {
                        String displayLanguage = LanguageUtil.getLanguage(strDestinationLanguage).
                                getLocale().getDisplayLanguage();
                        content.setProperty(JcrConstants.JCR_TITLE, displayLanguage);
                    }

                    //set JCR_TITLE for all folders in the created contentPath and language root
                    String tempAddTitlePath = contentPath.substring(0, contentPath.lastIndexOf('/'));
                    while (tempAddTitlePath.length() > 0) {
                        Node tempNode = session.getNode(languageRootPath + tempAddTitlePath);
                        if (tempNode != null) {
                            if (!tempNode.hasNode(JcrConstants.JCR_CONTENT)) {
                                tempNode.addNode(JcrConstants.JCR_CONTENT, JcrConstants.NT_UNSTRUCTURED);
                            }
                            Node tempNodeContent = tempNode.getNode(JcrConstants.JCR_CONTENT);
                            String sourceNodeForTitle = null;
                            if (createNewLanguageRoot) {
                                sourceNodeForTitle = parentOfRoot + tempAddTitlePath;
                            } else {
                                sourceNodeForTitle = root + tempAddTitlePath;
                            }
                            if (!tempNodeContent.hasProperty(JcrConstants.JCR_TITLE)) {
                                String tempNodeTitle = UIHelper.getTitle(resourceResolver.
                                        getResource(sourceNodeForTitle));
                                tempNodeContent.setProperty(JcrConstants.JCR_TITLE, tempNodeTitle);
                            }
                            tempAddTitlePath = tempAddTitlePath.substring(0, tempAddTitlePath.lastIndexOf('/'));
                        }
                    }
                }
            session.save();
        } catch (Exception e) {
            log.error(
                    "error while creating language copy for assets at path: "
                            + languageCopyPath + "{}", e);
        }

        return createdCopy;
    }

    private static String getDestinationLanguageWithAllowedDelimiters(String contentPath, String strDestinationLanguage,
        ResourceResolver resourceResolver) {
        String langWithHyphen = strDestinationLanguage.replace("_", "-");
        String langWithUnderscore = strDestinationLanguage.replace("-", "_");
        boolean langWithHyphenExist = (null != resourceResolver.getResource(contentPath + "/" + langWithHyphen));
        boolean langWithUnderscoreExist = (null != resourceResolver.getResource(contentPath + "/" +
            langWithUnderscore));

        if (langWithHyphenExist && langWithUnderscoreExist) {
            return strDestinationLanguage;
        } else if (langWithHyphenExist) {
            return langWithHyphen;
        } else if (langWithUnderscoreExist) {
            return langWithUnderscore;
        } else {
            return strDestinationLanguage;
        }
    }

    private static void deleteInsightData(String resourcePath, ResourceResolver resourceResolver) {
        Resource resource = resourceResolver.getResource(resourcePath);
        if (resource != null) {
            if (DamUtil.isAsset(resource)) {
                deleteResource(resourcePath + ASSET_PERFORMANCE_NODE_RELATIVE_PATH, resourceResolver);
                deleteResource(resourcePath + ASSET_USAGE_NODE_RELATIVE_PATH, resourceResolver);
            } else if (isFolder(resource)) {
                Iterator<Asset> assetIterator = DamUtil.getAssets(resource);
                while (assetIterator.hasNext()) {
                    String assetPath = assetIterator.next().getPath();
                    deleteResource(assetPath + ASSET_PERFORMANCE_NODE_RELATIVE_PATH, resourceResolver);
                    deleteResource(assetPath + ASSET_USAGE_NODE_RELATIVE_PATH, resourceResolver);
                }
            }
        }
    }

    private static void deleteResource(String path, ResourceResolver resourceResolver) {
        Resource resource = resourceResolver.getResource(path);
        if (resource != null) {
            try {
                resourceResolver.delete(resource);
            } catch (PersistenceException e) {
                log.error("Unable to delete resource from {} : {}", resource, e.getMessage());
            }
        }
    }


    private static void setDestinationLanguageCopyPath(Resource temporaryResource, String destinationPath)
            throws RepositoryException {
        Node temporaryNode = temporaryResource.adaptTo(Node.class);
        if (!temporaryNode.hasNode(JcrConstants.JCR_CONTENT)) {
            temporaryNode.addNode(JcrConstants.JCR_CONTENT, JcrConstants.NT_UNSTRUCTURED);
        }
        Node destinationContentNode = temporaryNode.getNode(JcrConstants.JCR_CONTENT);
        destinationContentNode.setProperty(ATTRIBUTE_DESTINATION_LANGUAGE_COPY_PATH, destinationPath);
    }

    private static boolean isFolder(Resource resource){
        Node n = resource.adaptTo(Node.class);
        try {
            return n.isNodeType("nt:folder");
        } catch (RepositoryException e) {
            return false;
        }
    }

    /**
     *
     * @param sourcePath
     * @param destinationPath
     * @param userSession
     * @param pageManagerFactory
     * @param resolverFactory
     *
     * @deprecated since 6.2, use
     * {@link #moveUpdatedAsset(String, String, Session, PageManagerFactory, ResourceResolver)}  instead
     */
    @Deprecated
    public static void moveUpdatedAsset(String sourcePath, String destinationPath, Session userSession,
                                        PageManagerFactory pageManagerFactory, ResourceResolverFactory resolverFactory){
        Map<String, Object> authInfo = new HashMap<String, Object>();
        authInfo.put(JcrResourceConstants.AUTHENTICATION_INFO_SESSION, userSession);
        ResourceResolver resourceResolver = null;
        try {
            resourceResolver = resolverFactory.getResourceResolver(authInfo);   ///todo check
            Resource sourceResource = resourceResolver.getResource(sourcePath);
            Resource destinationResource = resourceResolver.getResource(destinationPath);

            Asset sourceAsset = DamUtil.resolveToAsset(sourceResource);

            PageManager pageManager = pageManagerFactory.getPageManager(resourceResolver);
            if(destinationResource == null){
                String pathToCreate = Text.getRelativeParent(
                        destinationPath, 1);
                String nodeType = "sling:Folder";
//                if (userSession.getNode(destinationPath).isNodeType(
//                        "sling:OrderedFolder")) {
//                    nodeType = "sling:OrderedFolder";
//                }
                JcrResourceUtil.createPath(pathToCreate, nodeType,
                        nodeType, userSession, false);
                pageManager.copy(sourceResource, destinationPath, null, false,
                        false, true);
            } else {
                Asset destinationAsset = DamUtil.resolveToAsset(destinationResource);
                destinationAsset.addRendition(DamConstants.ORIGINAL_FILE, sourceAsset.getOriginal().getStream(),
                        sourceAsset.getMimeType());
                //todo copy metadata
            }

            userSession.save();
        } catch (Exception e) {
            log.error("Unable to move updated asset {}", e.getMessage());
        }
    }

    public static void moveUpdatedAsset(String sourcePath, String destinationPath, Session userSession,
                                        PageManagerFactory pageManagerFactory, ResourceResolver resourceResolver){
        Map<String, Object> authInfo = new HashMap<String, Object>();
        authInfo.put(JcrResourceConstants.AUTHENTICATION_INFO_SESSION, userSession);
        try {
            Resource sourceResource = resourceResolver.getResource(sourcePath);
            Resource destinationResource = resourceResolver.getResource(destinationPath);
            Asset sourceAsset = DamUtil.resolveToAsset(sourceResource);
            if(sourceAsset != null) {
                PageManager pageManager = pageManagerFactory.getPageManager(resourceResolver);
                if(destinationResource == null){
                    String pathToCreate = Text.getRelativeParent(
                            destinationPath, 1);
                    String nodeType = "sling:Folder";
//                if (userSession.getNode(destinationPath).isNodeType(
//                        "sling:OrderedFolder")) {
//                    nodeType = "sling:OrderedFolder";
//                }
                    JcrResourceUtil.createPath(pathToCreate, nodeType,
                            nodeType, userSession, false);
                    pageManager.copy(sourceResource, destinationPath, null, false,
                            false, true);
                } else {
                    Asset destinationAsset = DamUtil.resolveToAsset(destinationResource);
                    if(destinationAsset != null) {
                        destinationAsset.addRendition(DamConstants.ORIGINAL_FILE, sourceAsset.getOriginal().getStream(),
                                sourceAsset.getMimeType());
                    } else {
                        log.error("Unable to move updated asset : Destination Asset not found");
                    }
                    //todo copy metadata
                }
                userSession.save();
            } else {
                log.error("Unable to move updated asset : Source Asset not found");
            }
        } catch (Exception e) {
            log.error("Unable to move updated asset {}", e.getMessage());
        }
    }

    public static void replaceUpdatedAsset(String sourcePath, String destinationPath, Session userSession,
        PageManagerFactory pageManagerFactory, ResourceResolver resourceResolver){
        try {
            Resource sourceResource = resourceResolver.getResource(sourcePath);
            Resource destinationResource = resourceResolver.getResource(destinationPath);

            Asset sourceAsset = sourceResource.adaptTo(Asset.class);
            if (sourceAsset!= null && !isContentFragment(sourceAsset)) {
                addExtractMetadataPropertyForAsset(sourceAsset, false);
            }

            if(sourceResource != null) {
                PageManager pageManager = pageManagerFactory.getPageManager(resourceResolver);
                if(destinationResource == null){
                    String pathToCreate = Text.getRelativeParent(
                            destinationPath, 1);
                    String nodeType = "sling:Folder";
                    JcrResourceUtil.createPath(pathToCreate, nodeType,
                            nodeType, userSession, false);
                    pageManager.copy(sourceResource, destinationPath, null, false,
                        false, true);
                } else {
                    //create version
                    Asset destinationAsset = resourceResolver.getResource(destinationPath).adaptTo(Asset.class);

                    if(destinationAsset != null) {
                        destinationAsset.createRevision("", ASSET_VERSION_MESSAGE);
                        copyInsightData(destinationResource.getPath(), sourceResource.getPath(), resourceResolver,
                            pageManager);
                        deleteAllChildren(destinationResource, userSession);
                        copyAllChildren(sourceResource, destinationResource, pageManager, userSession);
                    } else {
                        log.error("Unable to move updated asset : Destination is not an Asset");
                    }
                }

                userSession.save();
            } else {
                log.error("Unable to move updated asset : Source Resource not found");
            }
        } catch (Exception e) {
            log.error("Unable to move updated asset {}", e.getMessage());
        }
    }

    private static void copyInsightData(String sourcePath, String destinationPath, ResourceResolver resourceResolver,
        PageManager pageManager) {
        copyResource(sourcePath + ASSET_PERFORMANCE_NODE_RELATIVE_PATH,
            destinationPath + ASSET_PERFORMANCE_NODE_RELATIVE_PATH, resourceResolver, pageManager);
        copyResource(sourcePath + ASSET_USAGE_NODE_RELATIVE_PATH, destinationPath + ASSET_USAGE_NODE_RELATIVE_PATH,
            resourceResolver, pageManager);
    }

    private static void copyResource(String sourcePath, String destinationPath, ResourceResolver resourceResolver,
        PageManager pageManager) {
        Resource sourceResource = resourceResolver.getResource(sourcePath);
        if (sourceResource != null) {
            try {
                pageManager.copy(sourceResource, destinationPath, null, false, false, false);
            } catch (WCMException e) {
                log.error("Unable to copy resource from " + sourcePath + " to " + destinationPath + " : {}",
                    e.getMessage());
            }
        }
    }

    private static void copyAllChildren(Resource sourceResource, Resource destinationResource, PageManager pageManager,
        Session session) throws RepositoryException, WCMException {
        if (sourceResource != null && destinationResource != null) {
            Iterable<Resource> childList = sourceResource.getChildren();
            String destinationResourcePath = destinationResource.getPath();
            for (Resource child : childList) {
                String destinationChildPath = destinationResourcePath + "/" + child.getName();
                pageManager.copy(child, destinationChildPath, null, false, false, false);
            }
            session.save();
        }
    }

    private static void deleteAllChildren(Resource resource, Session session) throws RepositoryException {
        if (resource != null) {
            Iterable<Resource> childList = resource.getChildren();
            for (Resource child : childList) {
                Node childNode = child.adaptTo(Node.class);
                childNode.remove();
            }
        }
        session.save();
    }

    /**
     * Removes all asset renditions contained in the resource.
     * If resource is a folder, then all its folders are navigated recursively to delete asset renditions.
     * @param resource resource
     */
    private static void removeAllRenditionsInsideResource(Resource resource) throws RepositoryException {
        Iterator<Asset> assets = DamUtil.getAssets(resource);
        while (assets.hasNext()) {
            Asset asset = assets.next();
            if (!isContentFragment(asset)) {
                List<Rendition> renditions = asset.getRenditions();
                for (Rendition rendition : renditions) {
                    String name = rendition.getName();
                    if (!name.equals(AssetConstants.ORIGINAL_FILE)) {
                        asset.removeRendition(name);
                    }
                }
            }
        }
    }

    private static boolean isContentFragment(Asset asset) throws RepositoryException {
        Node assetNode = asset.adaptTo(Node.class);
        Node jcrNode = assetNode.getNode(JCR_CONTENT);
        if (jcrNode.hasProperty(CONTENT_FRAGMENT)) {
            return jcrNode.getProperty(CONTENT_FRAGMENT).getBoolean();
        }
        return false;
    }

    public static boolean isSmartAssetUpdateRequired(Asset sourceAsset, Asset destinationAsset) {
        if (sourceAsset == null || destinationAsset == null) {
            return false;
        }
        return sourceAsset.getLastModified() > destinationAsset.getLastModified();
    }

    @Deprecated
    /**
     *@deprecated since 6.2, use
     *{@link #addSmartAssetUpdateSource(Asset, String)}  instead
     */
    public static void addSmartAssetUpdateFlag(Asset destinationAsset) throws RepositoryException {
        Node assetNode = destinationAsset.adaptTo(Node.class);
        if (!assetNode.hasNode(JcrConstants.JCR_CONTENT)) {
            assetNode.addNode(JcrConstants.JCR_CONTENT, JcrConstants.NT_UNSTRUCTURED);
        }
        Node assetContentNode = assetNode.getNode(JcrConstants.JCR_CONTENT);
        assetContentNode.setProperty(ATTRIBUTE_ASSET_UPDATE_REQUIRED, true);
    }

    public static void addSmartAssetUpdateSource(Asset destinationAsset, String sourcePath) throws RepositoryException {
        if (destinationAsset == null || sourcePath == null) {
            return;
        }
        Node assetNode = destinationAsset.adaptTo(Node.class);
        if (!assetNode.hasNode(JcrConstants.JCR_CONTENT)) {
            assetNode.addNode(JcrConstants.JCR_CONTENT, JcrConstants.NT_UNSTRUCTURED);
        }
        Node assetContentNode = assetNode.getNode(JcrConstants.JCR_CONTENT);
        assetContentNode.setProperty(ATTRIBUTE_SMART_ASSET_UPDATE_SOURCE, sourcePath);
    }

    private static void addExtractMetadataPropertyForAsset(Asset asset, boolean bExtractMetadata) throws RepositoryException {
        Node assetNode = asset.adaptTo(Node.class);
        if (!assetNode.hasNode(JcrConstants.JCR_CONTENT)) {
            assetNode.addNode(JcrConstants.JCR_CONTENT, JcrConstants.NT_UNSTRUCTURED);
        }
        Node assetContentNode = assetNode.getNode(JcrConstants.JCR_CONTENT);
        assetContentNode.setProperty(ATTRIBUTE_EXTRACT_METADATA, bExtractMetadata);
    }

}
