package com.day.cq.dam.commons.util;

import java.util.Calendar;
import java.util.Iterator;

import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;
import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;

import com.day.cq.dam.api.Asset;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang.StringUtils;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.tenant.Tenant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.cq.commons.jcr.JcrConstants;
import com.day.cq.dam.api.s7dam.config.DynamicMediaServicesConfig;

/**
 * Utility helper class to facilitate access to Dynamic Media services cloud configuration.
 * There can be only a single definition of this service.
 */
public class DynamicMediaServicesConfigUtil {

    private static final Logger LOG = LoggerFactory.getLogger(DynamicMediaServicesConfigUtil.class);
    // Do not change without contacting S7 video proxy server engineering, the
    //  The key is coordinated with the DM Video Proxy service.
    private static final int REGISTRATION_EXPIRES_SECONDS = 3 * 60 * 60;
    private static byte[] key = {
            0x6f, 0x6e, 0x27, 0x74, 0x20, 0x66, 0x6f, 0x72, 0x67, 0x65, 0x74, 0x20, 0x74, 0x6f, 0x64, 0x72
    };

    /**
     * Helper method to determine if Dynamic Media Service support cloud config has been defined.
     *
     * @param resolver a {@link org.apache.sling.api.resource.ResourceResolver} to be used for the operation
     * @return {@code Boolean} indicating if Dynamic Media is enabled or not
     */
    public static boolean isDynamicMediaServiceConfigured(ResourceResolver resolver) {
        return  isDynamicMediaServiceConfigured(resolver, null);
    }

    /**
     * Helper method to determine if Dynamic Media Service support cloud config has been defined.
     *
     * @param resolver a {@link org.apache.sling.api.resource.ResourceResolver} to be used for the operation
     * @param assetPath - optional uploaded asset path, invalid value refers to the non-tenant-aware environment
     * @return {@code Boolean} indicating if Dynamic Media is enabled or not
     */
    public static boolean isDynamicMediaServiceConfigured(ResourceResolver resolver, String assetPath) {
        return  getDefaultConfig(resolver, assetPath) != null;
    }

    /**
     * the default definition of DynamicMediaServicesConfig
     * @param resourceResolver
     * @return  DynamicMediaServicesConfig or null if it is not completely defined (includes a registration ID)
     */
    public static DynamicMediaServicesConfig getDefaultConfig(ResourceResolver resourceResolver) {
        return getDefaultConfig(resourceResolver, null);
    }

    /**
     * the default definition of DynamicMediaServicesConfig
     * @param resourceResolver
     * @param assetPath - optional uploaded asset path, invalid value refers to the non-tenant-aware environment
     * @return  DynamicMediaServicesConfig or null if it is not completely defined (includes a registration ID)
     */
    public static DynamicMediaServicesConfig getDefaultConfig(ResourceResolver resourceResolver, String assetPath) {
        String suffix = "";  // append to DM service CC path for tenant-aware environment
        if (assetPath != null) {
            Tenant tenant = getTenant(resourceResolver,assetPath);
            if (tenant != null) {
                String tenantId = tenant.getId();
                if (tenantId != null && tenantId.length() > 0) {
                    suffix = "/" + tenantId;
                }
            }
        }

        Resource cloudResource = resourceResolver.getResource(DynamicMediaServicesConfig.DM_SERVICES_CC_PATH + suffix);
        if (cloudResource != null) {
            Iterator<Resource> dmConfigResourceIter = cloudResource.listChildren();
            while (dmConfigResourceIter.hasNext()) {
                DynamicMediaServicesConfig config = dmConfigResourceIter.next().adaptTo(DynamicMediaServicesConfig.class);
                if (config != null && !StringUtils.isEmpty(config.getRegistrationId()) && !StringUtils.isEmpty(config.getServiceUrl())) {
                    return config;
                }
            }
        }
        return null;
    }


    /**
     * Get encrypted Registration Id for the Dynamic Media service
     * @param resourceResolver
     * @return  the registration Id or null if undefined
     *  This is an encrypted version of the registration
    */
    public static String getRegistrationId(ResourceResolver resourceResolver) {
        return getRegistrationId(resourceResolver, null);
    }
    /**
     * Get encrypted Registration Id for the Dynamic Media service
     * @param resourceResolver
     * @param assetPath - optional uploaded asset path, invalid value refers to the non-tenant-aware environment
     * @return  the registration Id or null if undefined
     *  This is an encrypted version of the registration
    */
    public static String getRegistrationId(ResourceResolver resourceResolver, String assetPath) {
        DynamicMediaServicesConfig config = DynamicMediaServicesConfigUtil.getDefaultConfig(resourceResolver, assetPath);
        Calendar cal = Calendar.getInstance();
        long registrationTimeout = cal.getTimeInMillis() + (REGISTRATION_EXPIRES_SECONDS * 1000);
        try {
            if (config != null) {
                return encrypt(config.getRegistrationId() + "|" + String.valueOf(registrationTimeout));
            }
        } catch (Exception e) {
            LOG.error("Failed while creating encrypted key [{}]", e.getMessage());
        }
        return null;
    }

    private static String encrypt(String strToEncrypt)
    {
        try
        {
            Cipher cipher = Cipher.getInstance("AES/ECB/PKCS5Padding");
            final SecretKeySpec secretKey = new SecretKeySpec(key, "AES");
            cipher.init(Cipher.ENCRYPT_MODE, secretKey);
            String enc = Base64.encodeBase64String(cipher.doFinal(strToEncrypt.getBytes()));

            // Tomcat server does not like slashes
            return enc.replaceAll("/", "_slash_").replaceAll("\\?", "_qmark_");
        }
        catch (Exception e)
        {
            LOG.error("Error while encrypting", e);
        }
        return null;

    }

    /**
     * The public key for the Dynamic Media service
     * @param resourceResolver
     * @return  the public key proxy server or null if undefined
     */
    public static String getPublicKey(ResourceResolver resourceResolver) {
        return getPublicKey(resourceResolver, null);
    }

    /**
     * The public key for the Dynamic Media service
     * @param resourceResolver
     * @param assetPath - optional uploaded asset path, invalid value refers to the non-tenant-aware environment
     * @return  the public key proxy server or null if undefined
     */
    public static String getPublicKey(ResourceResolver resourceResolver, String assetPath) {
        try {
            DynamicMediaServicesConfig config = DynamicMediaServicesConfigUtil.getDefaultConfig(resourceResolver, assetPath);
            if (config != null) {
                String regId = config.getRegistrationId();
                if (StringUtils.isNotBlank(regId) && regId.contains("|")) {
                    return StringUtils.substringAfter(regId, "|");
                }
            }
        } catch(Exception e){
            LOG.error("Failed while extracting public key [{}]", e.getMessage());
        }
        return null;
    }

    /**
     * The service url for the Dynamic Media pr0xy service
     * @param resourceResolver
     * @return  the service url or null if none exists
     */
    public static String getServiceUrl(ResourceResolver resourceResolver) {
        return getServiceUrl(resourceResolver, null);
    }

    /**
     * The service url for the Dynamic Media pr0xy service
     * @param resourceResolver
     * @param assetPath - optional uploaded asset path, invalid value refers to the non-tenant-aware environment
     * @return  the service url or null if none exists
     */
	public static String getServiceUrl(ResourceResolver resourceResolver, String assetPath) {
		String videoServiceUrl = null;

		// First try to get service url from dynamic media services config
		DynamicMediaServicesConfig config = DynamicMediaServicesConfigUtil
				.getDefaultConfig(resourceResolver, assetPath);
		if (config != null) {
			videoServiceUrl = config.getServiceUrl();
			if (videoServiceUrl.length() > 0) {
				return videoServiceUrl;
			}
		} else {
			// Now try for the default value
			try {
				String path = DynamicMediaServicesConfig.DM_SERVICES_CC_PATH
						+ "/" + JcrConstants.JCR_CONTENT;
				Resource res = resourceResolver.getResource(path);
				Node node = res.adaptTo(Node.class);
				if (node.hasProperty(DynamicMediaServicesConfig.PN_SERVICE_URL)) {
					videoServiceUrl = node.getProperty(
							DynamicMediaServicesConfig.PN_SERVICE_URL)
							.getString();
				}
			} catch (PathNotFoundException e) {
				LOG.error("videoServiceUrl access error {}", e);
			} catch (RepositoryException e) {
				LOG.error("videoServiceUrl access error {}", e);
			} catch (Exception e) {
				LOG.error("videoServiceUrl access error {}", e);
			}
		}

		return videoServiceUrl;
	}

    /**
     * The image service url for the Dynamic Media image delivery
     * @param resourceResolver Resource resolver
     * @return  the service url or empty if none exists
     */
    public static String getImageServiceUrl(ResourceResolver resourceResolver) {
        return getImageServiceUrl(resourceResolver, null);
    }

    /**
     * The image service url for the Dynamic Media image delivery
     * @param resourceResolver Resource resolver
     * @param assetPath - optional uploaded asset path, invalid value refers to the non-tenant-aware environment
     * @return  the service url or empty if none exists
     */
    public static String getImageServiceUrl(ResourceResolver resourceResolver, String assetPath) {
        String imageServiceUrl = "";
        DynamicMediaServicesConfig config = DynamicMediaServicesConfigUtil.getDefaultConfig(resourceResolver,assetPath);
        if (config != null) {
            imageServiceUrl = config.getImageServiceUrl();
        }
        return imageServiceUrl;
    }

    private static Tenant getTenant(ResourceResolver resourceResolver, String assetPath) {
        if (assetPath != null) {
            Resource assetResource = resourceResolver.getResource(assetPath);
            return assetResource.adaptTo(Tenant.class);
        }
        return null;
    }

}
