/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.dam.commons.util;

import org.apache.commons.lang.StringUtils;

import com.day.cq.dam.api.Asset;
import com.day.image.Layer;

/**
 * The <code>OrientationUtil</code> helps in adjusting a {@link com.day.image.Layer}
 * to the orientation stored in the image's EXIF or TIFF metadata.
 */
public class OrientationUtil {
    public static final short ORIENTATION_NORMAL = 1;
    public static final short ORIENTATION_MIRROR_HORIZONTAL = 2;
    public static final short ORIENTATION_ROTATE_180 = 3;
    public static final short ORIENTATION_MIRROR_VERTICAL = 4;
    public static final short ORIENTATION_MIRROR_HORIZONTAL_ROTATE_270_CW = 5;
    public static final short ORIENTATION_ROTATE_90_CW = 6;
    public static final short ORIENTATION_MIRROR_HORIZONTAL_ROTATE_90_CW = 7;
    public static final short ORIENTATION_ROTATE_270_CW = 8;
    
    /** EXIF orientation property */
    static final String EXIF_ORIENTATION = "exif:Orientation";
    /** TIFF orientation property */
    static final String TIFF_ORIENTATION = "tiff:Orientation";

    /**
     * Check if asset contains metadata for orientation.
     * @param asset The asset
     * @return      <code>true</code> if orientation metadata exists
     */
    public static boolean hasOrientationMetadata(Asset asset) {
        return (StringUtils.isNotEmpty(asset.getMetadataValue(EXIF_ORIENTATION))
                || StringUtils.isNotEmpty(asset.getMetadataValue(TIFF_ORIENTATION)));
    }

    /**
     * Get the exif orientation flag from the asset metadata.
     * @param asset the asset
     * @return the EXIF orientation flag (1-8)
     */
    public static short getOrientation(Asset asset) {
        // tiff:Orientation is the standard property for this flag for all image types,
        // not only TIFF, including EXIF data from JPEG files (see CQ-13795)
        String value = asset.getMetadataValue(TIFF_ORIENTATION);

        if (StringUtils.isEmpty(value)) {
            // fallback to an older exif:Orientation property
            value = asset.getMetadataValue(EXIF_ORIENTATION);
        }

        if (StringUtils.isEmpty(value)) {
            return ORIENTATION_NORMAL;
        }

        try {
            return Short.parseShort(value);
        } catch (NumberFormatException ne) {
            return ORIENTATION_NORMAL;
        }
    }
    
    /**
     * Adjust the given layer to the orientation stored in the asset's metadata.
     * @param asset The asset
     * @param layer The layer
     */
    public static void adjustOrientation(Asset asset, Layer layer) {
        switch (getOrientation(asset)) {
            case ORIENTATION_MIRROR_HORIZONTAL:
                layer.flipHorizontally();
                break;
            case ORIENTATION_ROTATE_180:
                layer.rotate(180);
                break;
            case ORIENTATION_MIRROR_VERTICAL:
                layer.flipVertically();
                break;
            case ORIENTATION_MIRROR_HORIZONTAL_ROTATE_270_CW:
                layer.flipHorizontally();
                layer.rotate(270);
                break;
            case ORIENTATION_ROTATE_90_CW:
                layer.rotate(90);
                break;
            case ORIENTATION_MIRROR_HORIZONTAL_ROTATE_90_CW:
                layer.flipHorizontally();
                layer.rotate(90);
                break;
            case ORIENTATION_ROTATE_270_CW:
                layer.rotate(270);
                break;
        }
    }
}
