/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.day.cq.dam.commons.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import javax.jcr.RepositoryException;

import com.adobe.granite.confmgr.Conf;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.jcr.resource.JcrResourceConstants;
import org.apache.sling.tenant.Tenant;

import com.day.cq.commons.jcr.JcrConstants;
import com.day.cq.dam.api.DamConstants;
import com.day.cq.dam.commons.schemaforms.internal.TabList;
import com.day.text.Text;

/**
 * this is an internal class to be used by dam code.
 */
public class SchemaFormHelper {

    /**
     * Given current form, returns a list of master.
     * Looks for master forms up in the path heirarchy,
     * while traversing it takes into account any override
     * available in apps
     *
     * @param currentForm current form resource
     * @return list of master form resource, empty list if no master is found
     * @throws RepositoryException
     */
    @Deprecated
    public static List<Resource> getMasterForms(Resource currentForm, String formsBaseDirPath) throws RepositoryException {
        final ResourceResolver resolver = currentForm.getResourceResolver();
        final String currentResourcePath = currentForm.getPath();

        final String appsDIr = getAppsDir(resolver);
        
        final String APPS_BASE_DIR = appsDIr + "/" + formsBaseDirPath;
        final String APPS_BASE_DIR_NON_TENANT = DamConstants.APPS + "/" + formsBaseDirPath;
        final String LIBS_BASE_DIR = DamConstants.LIBS + "/" + formsBaseDirPath;

        List<Resource> masterFormsList = new ArrayList<Resource>();
        String path = currentResourcePath;
        String relativePath;
        Resource res = currentForm;
        Resource resToVerify;

        for (; !path.equals(APPS_BASE_DIR) && !path.equals(LIBS_BASE_DIR) && !path.equals(APPS_BASE_DIR_NON_TENANT); ) {
            res = res.getParent();
            path = res.getPath();
            
            if (path.startsWith(DamConstants.LIBS)) {
            	relativePath = path.substring(DamConstants.LIBS.length());
            } else {
                if (path.startsWith(appsDIr)) {
                    relativePath = path.substring(appsDIr.length());
                } else {
                    relativePath = path.substring(DamConstants.APPS.length());
                }
            }
            
            // check if the res is coming from /apps
            // i.e. res is under /apps and contains overridden tabs
            resToVerify = resolver.getResource(appsDIr + relativePath);
            if (resToVerify != null && resToVerify.getChild("items/tabs") != null) {
                masterFormsList.add(resToVerify);
            } else {
                resToVerify = resolver.getResource(DamConstants.APPS + relativePath);
                if (resToVerify != null && resToVerify.getChild("items/tabs") != null) {
                    masterFormsList.add(resToVerify);
                } else {
                    resToVerify = resolver.getResource(DamConstants.LIBS + relativePath);
                    if (resToVerify != null) {
                        masterFormsList.add(resToVerify);
                    }
                }
            }
        }

        Collections.reverse(masterFormsList);

        return masterFormsList;
    }

    /**
     * @return Metadata Schema provides a OOTB form. And User can overlay this form as per use case.
     * It returns list of form path homes (Overlay, OOTB) in the order. e.g. [/conf/metdataschema, /libs/metadataschema]
     */
    public static String[] getBaseFormPaths(ResourceResolver resourceResolver){
        Tenant tenant = resourceResolver.adaptTo(Tenant.class);
        String schemaExtHome = null != tenant? (String)tenant.getProperty(DamConfigurationConstants.METADATA_SCHEMA_HOME) : DamConfigurationConstants.DEFAULT_METADATA_SCHEMA_HOME;
        String schemaHome = DamConfigurationConstants.OOTB_METADATA_SCHEMA_FORM_HOME;
        Resource schemaExtHomeRes = resourceResolver.getResource(schemaExtHome);
        if(null == schemaExtHome || schemaExtHome.trim().isEmpty() || schemaExtHomeRes == null){
            schemaExtHome = DamConfigurationConstants.DEFAULT_METADATA_SCHEMA_HOME;
        }else{
            Conf conf = schemaExtHomeRes.adaptTo(Conf.class);
            schemaHome = conf.getItem(DamConfigurationConstants.ADMIN_UI_OOTB_CONF_RELPATH).get(DamConfigurationConstants.METADATA_SCHEMA_HOME, DamConfigurationConstants.OOTB_METADATA_SCHEMA_FORM_HOME);
        }
        return new String[]{schemaExtHome, schemaHome};
    }

    /**
     * retruns relatibe path of the form '/default/image'
     * @param formPath
     * @param baseFormPaths
     * @return
     */
    private static String getRelativeFormPath(String formPath, String[] baseFormPaths){
        for(String baseFormPath : baseFormPaths){
            if (formPath.startsWith(baseFormPath)){
                String relativePath = formPath.substring(baseFormPath.length());
                relativePath = relativePath.startsWith("/") ? relativePath : relativePath + "/";
                relativePath = relativePath.endsWith("/") ? relativePath.substring(relativePath.length() -1 ) : relativePath;
                return relativePath;
            }
        }
        return formPath;
    }

    /**
     * Given current form, returns a list of master.
     * Looks for master forms up in the path heirarchy,
     * while traversing it takes into account any override
     * available in apps
     *
     * @param currentForm current form resource
     * @return list of master form resource, empty list if no master is found
     * @throws RepositoryException
     * @throws java.lang.NullPointerException
     */
    public static List<Resource> getMasterForms(Resource currentForm) throws RepositoryException {
        ResourceResolver resourceResolver = currentForm.getResourceResolver();
        String[] baseFormPaths = getBaseFormPaths(resourceResolver);
        String relativePath = getRelativeFormPath(currentForm.getPath(), baseFormPaths);
        List<Resource> masterFormsList = new ArrayList<Resource>();
        relativePath = relativePath.trim().isEmpty() ? "" : relativePath.substring(0, relativePath.lastIndexOf('/'));

        while(!relativePath.trim().isEmpty()){
            for(int i=0; i<baseFormPaths.length; i++){
                String baseFormPath = baseFormPaths[i];
                Resource resToVerify = resourceResolver.getResource(baseFormPath + relativePath);
                if(resToVerify != null && (i == baseFormPaths.length -1 ||  resToVerify.getChild("items/tabs") != null)){
                    masterFormsList.add(resToVerify);
                    break;
                }
            }
            relativePath = relativePath.trim().isEmpty() ? "" : relativePath.substring(0, relativePath.lastIndexOf('/'));
        }

        Collections.reverse(masterFormsList);
        return masterFormsList;
    }



    /**
     * Merge two list of tabs i.e. form/items/tabs
     *
     * @param oneTabList   resource representing one tab list
     * @param otherTabList resource representing other tab list
     * @return merged tab list resource
     */
    public static Resource mergeFormTabResource(Resource oneTabList, Resource otherTabList) {
        if (oneTabList == null) {
            return otherTabList;
        }

        if (otherTabList == null) {
            return oneTabList;
        }

        TabList aTab = new TabList(oneTabList);
        TabList oTab = new TabList(otherTabList);

        aTab.merge(oTab);

        return aTab;
    }

    public static Resource getSchemaResource(SlingHttpServletRequest request) {
        String suffix = request.getRequestPathInfo().getSuffix();
        suffix = suffix == null ? "" : suffix;
        ResourceResolver resourceResolver = request.getResourceResolver();
        String[] baseFormPaths = getBaseFormPaths(resourceResolver);
        for(String baseFormPath : baseFormPaths){
            Resource res = resourceResolver.getResource(baseFormPath + suffix);
            if(null != res){
                return res;
            }
        }
        return null;
    }

    private static String getAppsDir(ResourceResolver resolver) {
    	String appsDir = DamConstants.APPS;
    	
    	// Tenantification of apps directory
    	Tenant tenant = resolver.adaptTo(Tenant.class);
    	if (tenant != null) {
    		appsDir = appsDir + "/" + tenant.getId();
    	}
    	
    	return appsDir;
    }

    public static Iterator getSchemaFormsIterator(SlingHttpServletRequest request, int rows, int offset) {
        // rows and offset will be required in case we want infinite scrolling in the future

        List<Resource> resourceList = new ArrayList<Resource>();

        ResourceResolver resolver = request.getResourceResolver();
        String[] baseFormPaths = getBaseFormPaths(resolver);

        String suffix = request.getRequestPathInfo().getSuffix();

        suffix = null == suffix ? "" : suffix;
        suffix = getRelativeFormPath(suffix, baseFormPaths);
        //create a concatenated tree out of APPS_DIR & LIBS_DIR
        String appsResPath = baseFormPaths[0] + suffix;
        String libsResPath = baseFormPaths[1] + suffix;

        Resource appRes = resolver.getResource(appsResPath);
        Resource libsRes = resolver.getResource(libsResPath);

        // fill all the resources in /apps which are not in /libs
        if (appRes != null) {
            for (Iterator<Resource> appIter = appRes.listChildren(); appIter.hasNext(); ) {
                Resource ar = appIter.next();
                String resName = Text.getName(ar.getPath());

                if (libsRes == null || libsRes.getChild(resName) == null) {
                    if (ar.isResourceType(JcrConstants.NT_FOLDER) ||
                            ar.isResourceType(JcrResourceConstants.NT_SLING_ORDERED_FOLDER) ||
                            ar.isResourceType(JcrResourceConstants.NT_SLING_FOLDER))
                        resourceList.add(ar);
                }
            }
        }

        // for all resources common in /apps and /libs, take those that are in /apps
        // else take those that are in /libs
        if (libsRes != null) {
            for (Iterator<Resource> libsIter = libsRes.listChildren(); libsIter.hasNext(); ) {
                Resource lr = libsIter.next();
                String resName = Text.getName(lr.getPath());

                if (appRes != null && appRes.getChild(resName) != null && appRes.getChild(resName).getChild("items/tabs") != null) {
                    Resource ar = resolver.getResource(appRes.getPath() + "/" + resName);
                    if (ar.isResourceType(JcrConstants.NT_FOLDER) ||
                            ar.isResourceType(JcrResourceConstants.NT_SLING_ORDERED_FOLDER) ||
                            ar.isResourceType(JcrResourceConstants.NT_SLING_FOLDER)) {
                        resourceList.add(ar);
                    }
                } else {
                    if (lr.getChild("items/tabs") != null && lr.isResourceType(JcrConstants.NT_FOLDER) ||
                            lr.isResourceType(JcrResourceConstants.NT_SLING_ORDERED_FOLDER) ||
                            lr.isResourceType(JcrResourceConstants.NT_SLING_FOLDER))
                        resourceList.add(lr);
                }
            }
        }
        return resourceList.iterator();
    }

}
