package com.day.cq.dam.commons.watermark;

import com.day.image.Layer;

/**
 *  A {@link Watermark} object represents the properties of a watermark effect, namely, <br/>
 *  <ul>
 *     <li>position     - relative to the underlying source image</li>
 *     <li>top, left    - y, x coordinates from top-left corner of underlying image</li>
 *     <li>orientation  - angle of rotation as measured from the x-axis in clockwise direction</li>
 *     <li>opacity      - transparency of the watermark</li>
 * </ul>
 * <br/>
 * The position can be specified as a {@link Location} or using top, left values.<br/>
 * If specified as a {@link Location}, it is transformed to x,y using image height,width
 * values at the time of application of the watermark.
 */
public abstract class Watermark {

    public static final Location DEFAULT_LOCATION = Location.CENTER;  // Just visible, non-distracting
    public static final double DEFAULT_ORIENTATION = 0d;                    // The angle of -
    public static final float DEFAULT_OPACITY = 0.2f;                       // Just visible, non-distracting

    /**
     * Location of the watermark relative to the underlying image<br/>
     * Five predefined positions
     * <br/>
     * <ul>
     *     <li>TOP_LEFT := top = 0, left = 0</li>
     *     <li>TOP_RIGHT := top = 0, left = _width</li>
     *     <li>BOTTOM_LEFT := top = _height, left = 0</li>
     *     <li>BOTTOM_RIGHT := top = _height, left = _width</li>
     *     <li>CENTER := top = _height/2, left = _width/2</li>
     * </ul>
     */
    private Location position = DEFAULT_LOCATION;
    private int top = -1;
    private int left = -1;

    private double orientation = DEFAULT_ORIENTATION;

    private float opacity = DEFAULT_OPACITY;

    protected Watermark() {}

    protected Watermark(Location position, double orientation, float opacity) {
        this.position = position;
        this.opacity = opacity;
        this.orientation = orientation;
    }

    protected Watermark(int top, int left, double orientation, float opacity) {
        this.top = top;
        this.left = left;
        this.orientation = orientation;
        this.opacity = opacity;
    }

    protected Watermark(Location position) {
        this.position = position;
    }

    protected Watermark(int top, int left) {
        this.top = top;
        this.left = left;
    }

    public Location getPosition() {
        return position;
    }

    public void setPosition(Location position) {
        this.position = position;
    }

    public int getTop() {
        return top;
    }

    public void setTop(int top) {
        this.top = top;
    }

    public int getLeft() {
        return left;
    }

    public void setLeft(int left) {
        this.left = left;
    }

    public double getOrientation() {
        return orientation;
    }

    public void setOrientation(int orientation) {
        this.orientation = orientation;
    }

    public float getOpacity() {
        return opacity;
    }

    public void setOpacity(float opacity) {
        this.opacity = opacity;
    }

    public void setCoords(int imgWidth, int imgHeight, int wmWidth, int wmHeight) {
        switch (this.position) {
            case TOP_LEFT:
                this.top=0;
                this.left=0;
                break;
            case BOTTOM_LEFT:
                this.top=imgHeight-wmHeight;
                this.left=0;
                break;
            case TOP_RIGHT:
                this.top=0;
                this.left=imgWidth-wmWidth;
                break;
            case BOTTOM_RIGHT:
                this.top=imgHeight-wmHeight;
                this.left=imgWidth-wmWidth;
                break;
            case CENTER:
                this.top=(imgHeight-wmHeight)/2;
                this.left=(imgWidth-wmWidth)/2;
                break;
        }
    }

    @Override
    public String toString() {
        return  "\ntop = " + top +
                "\nleft = " + left +
                "\norientation = " + orientation +
                "\nopacity = " + opacity +
                "\nposition = " + position.name();
    }
}
