/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.dam.core.process;

import static com.day.cq.commons.jcr.JcrConstants.JCR_CONTENT;
import static com.day.cq.dam.api.DamConstants.METADATA_FOLDER;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.PropertyIterator;
import javax.jcr.Session;
import javax.jcr.Value;

import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Service;
import org.apache.jackrabbit.JcrConstants;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.cq.dam.api.Asset;
import com.day.cq.dam.api.DamConstants;
import com.day.cq.dam.commons.process.AbstractAssetWorkflowProcess;
import com.day.cq.workflow.WorkflowException;
import com.day.cq.workflow.WorkflowSession;
import com.day.cq.workflow.exec.WorkItem;
import com.day.cq.workflow.metadata.MetaDataMap;

/**
 * The <code>ApplyMetadataTemplateProcess</code> is called in a workflow
 * process step. This process will apply metadata template to the asset, if the
 * paylod of the step is an {@link com.day.cq.dam.api.Asset Asset} or a part of
 * an {@link com.day.cq.dam.api.Asset Asset}.
 * 
 * @see AbstractAssetWorkflowProcess
 */
@Component(metatype = false)
@Service
@Property(name = "process.label", value = "Apply Metadata Template")
public class ApplyMetadataTemplateProcess extends AbstractAssetWorkflowProcess {

    /**
     * Logger instance for this class.
     */
    private static final Logger log = LoggerFactory.getLogger(ApplyMetadataTemplateProcess.class);

    private static final String JCR_CONTENT_METADATA = JCR_CONTENT + "/"
        + METADATA_FOLDER;

    public void execute(WorkItem workItem, WorkflowSession workflowSession,
            MetaDataMap args) throws WorkflowException {
        try {
            final Session session = workflowSession.getSession();
            final Asset asset = getAssetFromPayload(workItem, session);

            if (null != asset) {
                Node assetNode = asset.adaptTo(Node.class);
                Node node = assetNode;
                Node assetMetadataNode;

                boolean metadataTemplatePresent = false;
                while (!node.getPath().equals("/content") && !metadataTemplatePresent) {
                    node = node.getParent();
                    if (!node.getPrimaryNodeType().getName().equals(DamConstants.NT_DAM_ASSET) 
                    		&& node.hasNode(JCR_CONTENT_METADATA)) {
                        metadataTemplatePresent = true;
                        Node folderMetadataNode = node.getNode(JCR_CONTENT_METADATA);
                        if (!assetNode.hasNode(JCR_CONTENT_METADATA)) {
                            Node _jcr_content;
                            if (!assetNode.hasNode(JCR_CONTENT)) {
                                _jcr_content = assetNode.addNode(JCR_CONTENT,
                                    JcrConstants.NT_UNSTRUCTURED);
                            } else {
                                _jcr_content = assetNode.getNode(JCR_CONTENT);
                            }
                            assetMetadataNode = _jcr_content.addNode(
                                "metadata", JcrConstants.NT_UNSTRUCTURED);
                        } else {
                            assetMetadataNode = assetNode.getNode(JCR_CONTENT_METADATA);
                        }

                        PropertyIterator dpi = folderMetadataNode.getProperties();
                        while (dpi.hasNext()) {
                            javax.jcr.Property p = dpi.nextProperty();
                            if (!p.getName().startsWith("jcr:")) {
                                /**
                                 * There are cases where we have single valued
                                 * form items but the metadata information
                                 * extracted from the asset have multi valued
                                 * attribute for the same, so in a case of
                                 * multivalued default metadata value for the
                                 * asset, we append metadata vaules to list of
                                 * default values present in the asset.
                                 */
                                if (assetMetadataNode.hasProperty(p.getName())) {
                                    javax.jcr.Property prop = assetMetadataNode.getProperty(p.getName());
                                    Value[] vals;
                                    List<Value> updatedVals;
                                    /*
                                     * Value[] vals = prop.getValues();
                                     * List<Value> updatedVals = new
                                     * ArrayList<Value>();
                                     * updatedVals.addAll(Arrays.asList(vals));
                                     */

                                    if (p.isMultiple() && prop.isMultiple()) {
                                        vals = prop.getValues();
                                        updatedVals = new ArrayList<Value>();
                                        updatedVals.addAll(Arrays.asList(vals));
                                        for (Value val:p.getValues()){
                                        	if (!updatedVals.contains(val))
                                        		updatedVals.add(val);
                                        }
                                        assetMetadataNode.setProperty(
                                            prop.getName(),
                                            updatedVals.toArray(new Value[updatedVals.size()]),
                                            prop.getType());

                                    } else if (!p.isMultiple()
                                        && prop.isMultiple()) {
                                        vals = prop.getValues();
                                        updatedVals = new ArrayList<Value>();
                                        updatedVals.addAll(Arrays.asList(vals));
                                        if (!updatedVals.contains(p.getValue()))
                                    		updatedVals.add(p.getValue());
                                        assetMetadataNode.setProperty(
                                            prop.getName(),
                                            updatedVals.toArray(new Value[updatedVals.size()]),
                                            prop.getType());
                                    } else if (!p.isMultiple()
                                        && !prop.isMultiple()) {
                                    	assetMetadataNode.setProperty(
                                            p.getName(), p.getValue());
                                    }

                                    // TODO
                                    // Handle case when asset metadata has
                                    // single valued field
                                    // whereas the form item is multi valued
                                } else {
                                    if (!p.isMultiple()) {
                                        assetMetadataNode.setProperty(
                                            p.getName(), p.getValue());
                                    } else {
                                        assetMetadataNode.setProperty(
                                            p.getName(), p.getValues(),
                                            p.getType());
                                    }
                                }
                            }
                        }

                        // session.save need not be called here.
                        // It is automatically called on workflow session.

                        // session.save();
                    }
                }

            } else {
                String wfPayload = workItem.getWorkflowData().getPayload().toString();
                String message = "execute: cannot apply metadata template, asset [{"
                    + wfPayload
                    + "}] in payload doesn't exist for workflow [{"
                    + workItem.getId() + "}].";
                throw new WorkflowException(message);
            }
        } catch (Exception e) {
            log.warn(
                "unexpected error occurred during metadata extraction. Cause: {}",
                e.getMessage(), e);
        }
    }

}
