/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *************************************************************************/
package com.day.cq.dam.core.process;

import com.adobe.granite.offloading.api.OffloadingJobProperties;
import com.adobe.granite.offloading.workflow.api.WorkflowOffloadingHelper;
import com.adobe.granite.offloading.workflow.api.WorkflowOffloadingProcessArguments;
import com.adobe.granite.offloading.workflow.api.WorkflowOffloadingProperties;
import com.adobe.granite.workflow.WorkflowException;
import com.adobe.granite.workflow.WorkflowSession;
import com.adobe.granite.workflow.exec.WorkItem;
import com.adobe.granite.workflow.exec.WorkflowExternalProcess;
import com.adobe.granite.workflow.metadata.MetaDataMap;
import com.day.cq.dam.api.Asset;
import com.day.cq.dam.commons.util.DamUtil;
import org.apache.commons.lang.StringUtils;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.Service;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.api.wrappers.ValueMapDecorator;
import org.apache.sling.event.jobs.Job;
import org.apache.sling.event.jobs.JobManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Serializable;
import java.util.HashMap;

/**
 * This component implements an workflow external step, that allows offloading asset based workflows, means workflows
 * that have an asset on the payload.
 * <p/>
 * The workflow (model) to offload must be set on the step metadata using the <code>workflowModel</code> argument.
 * <p/>
 */
@Component
@Service
public class AssetOffloadingProcess implements WorkflowExternalProcess {

    private static final String TYPE_JCR_PATH = "JCR_PATH";

    private final Logger log = LoggerFactory.getLogger(getClass());

    /**
     * The available arguments to this process implementation.
     */
    public enum Arguments {
        INCLUDE_MIME_TYPES;
    }
    @Reference
    private JobManager jobManager;

    public Serializable execute(WorkItem item, WorkflowSession wfSession, MetaDataMap arguments) throws WorkflowException {
        String[]  includeMimeTypes = arguments.get(Arguments.INCLUDE_MIME_TYPES.name(),
            String[].class);
        
        Asset asset = getAsset(item, wfSession, arguments);
        
        if(!doProcess(asset,includeMimeTypes)) {
            log.debug("Skipping for asset "+asset.getPath());
            return null;
        }
        if (asset != null) {
            ValueMap jobProperties = getAllJobProperties(asset, item, arguments);
            String jobTopic = WorkflowOffloadingHelper.getJobTopic(arguments);
            if (StringUtils.isEmpty(jobTopic)) {
                throw new WorkflowException("job topic must not be empty");
            }
            Job offloadingJob = jobManager.addJob(jobTopic, jobProperties);
            if (offloadingJob == null) {
                throw new WorkflowException("not able to create job");
            }
            return offloadingJob.getId();
        } else {
            throw new WorkflowException("Can not execute step. Not a path based workflow payload");
        }

    }

    public boolean hasFinished(Serializable externalProcessId, WorkItem item, WorkflowSession session, MetaDataMap arguments) {
        Job offloadingJob = jobManager.getJobById((String) externalProcessId);
        return offloadingJob == null;
    }

    public void handleResult(Serializable externalProcessId, WorkItem workItem, WorkflowSession session, MetaDataMap arguments) throws WorkflowException {
        //nothing to do here...
    }

    protected Asset getAsset(WorkItem item, WorkflowSession wfSession, MetaDataMap arguments) {
        Asset asset = null;
        String payload = arguments.get(WorkflowOffloadingProcessArguments.WORKFLOW_PAYLOAD.getArgumentName(), String.class);
        if (StringUtils.isBlank(payload)) {
            if (item.getWorkflowData().getPayloadType().equals(TYPE_JCR_PATH)) {
                payload = item.getWorkflowData().getPayload().toString();
            }
        }
        final ResourceResolver resolver = wfSession.adaptTo(ResourceResolver.class);
        final Resource resource = resolver.getResource(payload);
        if (null != resource) {
            asset = DamUtil.resolveToAsset(resource);
        } else {
            log.error("getAsset: asset [{}] in payload of workflow [{}] does not exist.", payload,
                    item.getWorkflow().getId());
        }
        return asset;
    }

    private ValueMap getAllJobProperties(Asset asset, WorkItem item, MetaDataMap arguments) throws WorkflowException {
        ValueMap allJobProperties = new ValueMapDecorator(new HashMap<String, Object>());

        String workflowModel = WorkflowOffloadingHelper.getWorkflowModel(arguments);
        if (StringUtils.isEmpty(workflowModel)) {
            throw new WorkflowException("No workflow model specified, cannot execute workflow.");
        }
        String workflowPayload = asset.getOriginal().getPath();
    
        //adding workflow model and payload
        allJobProperties.put(WorkflowOffloadingProperties.OFFLOADING_WORKFLOW_MODEL.getPropertyName(), workflowModel);
        allJobProperties.put(WorkflowOffloadingProperties.OFFLOADING_WORKFLOW_PAYLOAD.getPropertyName(), workflowPayload);
        
        //add the generic job payload
        allJobProperties.putAll(WorkflowOffloadingHelper.getJobProperties(arguments));
        
         //add the offloading job payload
        String workflowOffloadingInput = WorkflowOffloadingHelper.getWorkflowOffloadingInput(arguments, asset.getPath(), workflowModel);
        allJobProperties.put(OffloadingJobProperties.INPUT_PAYLOAD.propertyName(), workflowOffloadingInput);
        String workflowOffloadingOutput = WorkflowOffloadingHelper.getWorkflowOffloadingOutput(arguments, asset.getPath(), workflowModel);
        allJobProperties.put(OffloadingJobProperties.OUTPUT_PAYLOAD.propertyName(), workflowOffloadingOutput);

        return allJobProperties;
    }
    
    private boolean doProcess(final Asset asset,final String[] includeMimeTypes) {
        if (asset == null) {
            return false;
        }
        if(includeMimeTypes == null) {
            return true;
        }
        String mimeType = asset.getMimeType();
        for (int i = 0; i < includeMimeTypes.length; i++) {
            if (includeMimeTypes[i].equalsIgnoreCase(mimeType) || includeMimeTypes[i].equals("*")) {
                return true;
            }
        }
        
        return false;

    }


}
