/*
 * Copyright 1997-2010 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.dam.core.process;

import com.day.cq.dam.api.Asset;
import com.day.cq.dam.api.DamConstants;
import com.day.cq.dam.api.Rendition;
import com.day.cq.dam.commons.process.AbstractAssetWorkflowProcess;
import com.day.cq.dam.core.impl.PrivateConstants;
import com.day.cq.workflow.WorkflowException;
import com.day.cq.workflow.WorkflowSession;
import com.day.cq.workflow.exec.WorkItem;
import com.day.cq.workflow.metadata.MetaDataMap;
import com.day.text.Text;
import org.apache.commons.lang.StringUtils;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Service;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import static com.day.cq.commons.jcr.JcrConstants.JCR_CONTENT;
import static com.day.cq.commons.jcr.JcrConstants.JCR_LASTMODIFIED;
import static com.day.cq.commons.jcr.JcrConstants.JCR_LAST_MODIFIED_BY;
import static com.day.cq.dam.api.DamConstants.DC_MODIFIED;
import static com.day.cq.dam.api.DamConstants.METADATA_FOLDER;

/**
 * The <code>AssetSetLastModifiedProcess</code> sets the last modified
 * properties of the {@link com.day.cq.dam.api.Asset Asset}. These are the
 * <i>jcr:lastModified</i> and <i>jcr:lastModifiedBy</i>
 * {@link javax.jcr.Property Properties} of Asset's Node and its MetaData <i>
 * {@value com.day.cq.dam.api.DamConstants#DC_MODIFIED}</i> value.
 * <p/>
 * In addition custom properties for the modification date and modifier can be
 * set.
 * <p/>
 * <b>Arguments:</b>
 * <table>
 * <thead>
 * <tr>
 * <td>Prefix</td>
 * <td>Description</td>
 * <td>Example</td>
 * </tr>
 * </thead>
 * <tr>
 * <td>relativeLastModifiedPath:</td>
 * <td>A path pointing to a {@link javax.jcr.Property Property} that may store a
 * {@link java.util.Calendar Calendar}. The Asset's Content Node will be take as
 * root for the relative Path</td>
 * <td>relativeLastModifiedPath:myModificationDate</td>
 * </tr>
 * <tr>
 * <td>relativeLastModifiedByPath:</td>
 * <td>A path pointing to a {@link javax.jcr.Property Property} that may store a
 * String. The Asset's Content Node will be take as root for the relative Path</td>
 * <td>relativeLastModifiedByPath:myModifier</td>
 * </tr>
 * </table>
 *
 * @see AbstractAssetWorkflowProcess
 */
@Component(metatype = false)
@Service
@Property(name = "process.label", value = "Set Last Modified")
public class AssetSetLastModifiedProcess extends AbstractAssetWorkflowProcess {

    /**
     * Logger instance for this class.
     */
    private static final Logger log = LoggerFactory.getLogger(AssetSetLastModifiedProcess.class);

    /**
     * The available arguments to this process implementation.
     */
    public enum Arguments {
        PROCESS_ARGS("PROCESS_ARGS"), RELATIVE_LAST_MODIFIED_PATH("relativeLastModifiedPath"), RELATIVE_LAST_MODIFIED_BY_PATH(
                "relativeLastModifiedByPath");
        private String argumentName;

        Arguments(String argumentName) {
            this.argumentName = argumentName;
        }

        public String getArgumentName() {
            return this.argumentName;
        }

        public String getArgumentPrefix() {
            return this.argumentName + ":";
        }
    }




    public void execute(WorkItem workItem, WorkflowSession workflowSession, MetaDataMap metaData) throws WorkflowException {
        String[] args = buildArguments(metaData);

        final Session session = workflowSession.getSession();
        final Asset asset = getAssetFromPayload(workItem, session);

        if (null != asset) {
            try {
                final Node assetNode = asset.adaptTo(Node.class);

                // get user id
                String userId = workItem.getWorkflowData().getMetaDataMap().get("userId", String.class);
                
                final Node content = assetNode.getNode(JCR_CONTENT);
                final Node metadata = assetNode.getNode( JCR_CONTENT + "/" + METADATA_FOLDER);
                String resolvedUser = userId;
                
                String payloadPath=null;
                
                if (workItem.getWorkflowData().getPayloadType().equals(TYPE_JCR_PATH)) {
                    payloadPath = workItem.getWorkflowData().getPayload().toString();
                }
                
                if(null !=payloadPath)
                {
                    final Node payloadNode = session.getNode(payloadPath);
                    // cases of nested child nodes. Parent would have already taken care of this.
                    if(payloadNode.hasProperty(PrivateConstants.SYNC_FLAG) )
                    {
                        return;
                    }
                }
                // this is to identify if this got trigerred because of Update
                // Asset workflow, in case of new rendition or update to orignal.
                if (content.hasProperty(PrivateConstants.SYNC_FLAG)) {
                    Rendition rendition = asset.getRendition(DamConstants.ORIGINAL_FILE);
                    if (rendition != null) {
                        String lastModified = (String) rendition.getProperties().get(
                            JCR_LAST_MODIFIED_BY);
                        
                        //It can be blank in cases where there was no last modified for the original
                        if (StringUtils.isNotBlank(lastModified)) {
                            resolvedUser = lastModified;
                            
                            content.setProperty(JCR_LAST_MODIFIED_BY, resolvedUser);
                            content.setProperty(JCR_LASTMODIFIED, Calendar.getInstance());
                        }

                        //we can return here as in case of new rendition, we need not to update last modified for metadata node
                        return;
                    }
                } 
                
                // set jcr:lastModified
                List<String> relLastModifiedPath = getValuesFromArgs("relativeLastModifiedPath", args);
                if (relLastModifiedPath.size() > 0) {
                    String nodePath = Text.getRelativeParent(asset.getPath() + "/" + JCR_CONTENT + "/"
                            + relLastModifiedPath.get(0), 1);
                    Node node = (Node) session.getItem(nodePath);
                    node.setProperty(Text.getName(relLastModifiedPath.get(0)), Calendar.getInstance());
                }
                // set jcr:lastModified
                List<String> relLastModifiedByPath = getValuesFromArgs("relativeLastModifiedByPath", args);
                if (relLastModifiedByPath.size() > 0) {
                    String nodePath = Text.getRelativeParent(asset.getPath() + "/" + JCR_CONTENT + "/"
                            + relLastModifiedByPath.get(0), 1);
                    Node node = (Node) session.getItem(nodePath);
                    node.setProperty(Text.getName(relLastModifiedByPath.get(0)), resolvedUser);
                }
                
                // update always
                content.setProperty(JCR_LAST_MODIFIED_BY, resolvedUser);
                content.setProperty(JCR_LASTMODIFIED, Calendar.getInstance());
                
                // set dc:modified only if not yet set
                String dcModifiedPath = JCR_CONTENT + "/" + METADATA_FOLDER + "/" + DC_MODIFIED;
                if (!assetNode.hasProperty(dcModifiedPath)) {
                    assetNode.getNode(JCR_CONTENT + "/" + METADATA_FOLDER).setProperty(DC_MODIFIED,
                            Calendar.getInstance());
                }
            } catch (RepositoryException e) {
                log.error("execute: repository error while setting last modified for asset [{}] in workflow ["
                        + workItem.getId() + "]", asset.getPath(), e);
            }

        } else {
			String wfPayload = workItem.getWorkflowData().getPayload().toString();
			String message = "execute: cannot set last modified, asset [{"+ wfPayload + "}] in payload doesn't exist for workflow [{"+ workItem.getId() + "}].";
			throw new WorkflowException(message);
            
        }
    }

    public String[] buildArguments(MetaDataMap metaData) {

        // the 'old' way, ensures backward compatibility
        String processArgs = metaData.get(Arguments.PROCESS_ARGS.name(), String.class);
        if (processArgs != null && !processArgs.equals("")) {
            return processArgs.split(",");
        }

        else {
            List<String> arguments = new ArrayList<String>();

            String relativeLastModifiedPath = metaData.get(Arguments.RELATIVE_LAST_MODIFIED_PATH.name(), String.class);
            String relativeLastModifiedByPath = metaData.get(Arguments.RELATIVE_LAST_MODIFIED_BY_PATH.name(),
                    String.class);

            if (StringUtils.isNotBlank(relativeLastModifiedPath)) {
                arguments.add(Arguments.RELATIVE_LAST_MODIFIED_PATH.getArgumentPrefix() + relativeLastModifiedPath);
            }

            if (StringUtils.isNotBlank(relativeLastModifiedByPath)) {
                arguments
                        .add(Arguments.RELATIVE_LAST_MODIFIED_BY_PATH.getArgumentPrefix() + relativeLastModifiedByPath);
            }

            return arguments.toArray(new String[arguments.size()]);
        }
    }
}
