/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.dam.core.process;

import aQute.bnd.annotation.ProviderType;

import com.day.cq.dam.api.Asset;
import com.day.cq.dam.api.Rendition;
import com.day.cq.dam.api.handler.AssetHandler;
import com.day.cq.dam.api.renditions.RenditionMaker;
import com.day.cq.dam.api.renditions.RenditionTemplate;
import com.day.cq.dam.api.thumbnail.ThumbnailConfig;
import com.day.cq.dam.commons.process.AbstractAssetWorkflowProcess;
import com.day.cq.dam.commons.thumbnail.ThumbnailGenerator;
import com.day.cq.dam.core.impl.CommandLineProcessTimeOutException;
import com.day.cq.workflow.WorkflowException;
import com.day.cq.workflow.WorkflowSession;
import com.day.cq.workflow.exec.WorkItem;
import com.day.cq.workflow.metadata.MetaDataMap;

import org.apache.commons.exec.CommandLine;
import org.apache.commons.exec.DefaultExecutor;
import org.apache.commons.exec.ExecuteException;
import org.apache.commons.exec.ExecuteWatchdog;
import org.apache.commons.exec.PumpStreamHandler;
import org.apache.commons.exec.ShutdownHookProcessDestroyer;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.Service;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.jcr.RepositoryException;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Workflow process that calls a command-line program and uses each output file
 * that it produces as an additional rendition. Optionally creates thumbnails
 * based on those renditions.<br>
 * The dimensions given in the thumbnail specifications are the maximal size the
 * result must have. Aspect ratio is maintained for image resizing. As a
 * precondition, the payload of this Workflow process has to be an
 * {@link com.day.cq.dam.api.Asset Asset} or part of an Asset.
 * <p>
 * <p/>
 * Example with the following workflow step arguments:
 *
 * <pre>
 *    mime:application/postscript,
 *    tn:140:100,tn:48:48,
 *    cmd:/bin/convert ${directory}/${filename} ${directory}/${basename}.jpg
 * </pre>
 * <p/>
 * The process will call /bin/convert, pass it the full path of the asset being
 * processed (temporarily dumped to disk), and create thumbnails of size 140x100
 * and 48x48 based on the output created by /bin/convert. <br>
 * This will only happen for assets having the
 * <code>application/postscript</code> mime-type, others are ignored.
 * <p/>
 * <b>Arguments:</b>
 * <table>
 * <thead>
 * <tr>
 * <td>Name</td>
 * <td>Prefix</td>
 * <td>Description</td>
 * <td>Required</td>
 * <td>Multiple</td>
 * <td>Example</td>
 * </tr>
 * </thead>
 * <tr>
 * <td>Command</td>
 * <td>cmd:</td>
 * <td>Command as executed on the consle.<br>
 * The command can contain varibales which are replaced before execution. The
 * following variables are available:<br>
 * <i>filename</i>: name of the file as exported to disk.<br>
 * <i>file</i>: absolute path of the file exported.<br>
 * <i>directory</i>: absolute path of the directory the command is run and the
 * asset is exported to.<br>
 * <i>basename</i>: the assets name on the disk without possible file-extension<br>
 * <i>extension</i>: the assets file extension.<br>
 * </td>
 * <td>required</td>
 * <td>multiple</td>
 * <td>cmd:/bin/convert ${directory}/${filename} ${directory}/${basename}.jpg</td>
 * </tr>
 * <tr>
 * <td>Mimetype Filter</td>
 * <td>mime:</td>
 * <td>Mimetype the Asset must have. If the Asset is of a diffrent type, this
 * process will not be executed.</td>
 * <td>required</td>
 * <td>multiple</td>
 * <td>mime:application/postscript</td>
 * </tr>
 * <tr>
 * <td>Thumbnail Specification</td>
 * <td>tn:</td>
 * <td>Dimensions of the Thumbnails to be generated from the Asset.<br>
 * <i>width</i>:<i>height</i>[:<i>false</i>]<br>
 * <i>width</i>:Number the maximal width in Pixel the Thumbnail must not exceed
 * <i>height</i>:Number the maximal height in Pixel the Thumbnail must not
 * exceed. <i>center</i>:Boolean, optional flag to indicate that the resulting
 * Thumbnail must not be centered, true is default.</td>
 * <td></td>
 * <td>multiple</td>
 * <td>tn:140:100</td>
 * </tr>
 * </table>
 *
 * @see AbstractAssetWorkflowProcess
 */
@Component(metatype = false)
@Service
@Property(name = "process.label", value = "Command Line")
@ProviderType
public class CommandLineProcess extends AbstractAssetWorkflowProcess {

    private final Logger log = LoggerFactory.getLogger(getClass());

    /**
     * The available arguments to this process implementation.
     */
    public enum Arguments {
        PROCESS_ARGS("PROCESS_ARGS"),

        /** Argument for the mime types to use */
        MIME_TYPES("mime"),

        /** Argument for thumb nail sizes */
        THUMBNAILS("tn"),

        /** Argument for command-line to execute */
        COMMANDS("cmd"),

        /** Argument to generate web renditions **/
        GENERATE_WEB_RENDITION("genWebRendition"),

        /** Argument to delete rendition generated by command prompt **/
        DELETE_COMMAND_RENDITION("deleteCommandRendition"),

        /** Argument for timeout */
        TIMEOUT("timeOut"),

        /** Argument for the number of retries */
        RETRIES("retries");

        private String argumentName;

        Arguments(String argumentName) {
            this.argumentName = argumentName;
        }

        public String getArgumentName() {
            return this.argumentName;
        }

        public String getArgumentPrefix() {
            return this.argumentName + ":";
        }

    }

    @Reference
    protected RenditionMaker renditionMaker;

    private static final String defaultCommandLineProcessTimeOut = "300000"; //5 minutes

    private static final String defaultNumberOfRetries = "0"; //0 retries

    private CreateWebEnabledImageProcess webEnabledImageCreator = new CreateWebEnabledImageProcess();

    public void execute(WorkItem workItem, WorkflowSession wfsession, MetaDataMap args) throws WorkflowException {
        String[] arguments = buildArguments(args);
        String input = args.get(Arguments.TIMEOUT.name(),String.class);
        final int commandLineProcessTimeOut = StringUtils.isNotBlank(input) ? Integer.parseInt(input) : Integer.parseInt(defaultCommandLineProcessTimeOut);
        input = args.get(Arguments.RETRIES.name(),String.class);
        final int numberOfRetries = StringUtils.isNotBlank(input) ? Integer.parseInt(input) : Integer.parseInt(defaultNumberOfRetries);

        final Asset asset = getAssetFromPayload(workItem, wfsession.getSession());

        File tmpDir = null;
        InputStream is = null;
        OutputStream os = null;
        try {
            // Process only specific mime types, based on arguments
            final List<String> mimeTypes = new LinkedList<String>();
            final String assetMimeType = asset.getMimeType();
            for (String str : arguments) {
                if (str.startsWith(Arguments.MIME_TYPES.getArgumentPrefix())) {
                    final String mt = str.substring(Arguments.MIME_TYPES.getArgumentPrefix().length()).trim();
                    log.debug("execute: accepted mime type [{}] for asset [{}].", mt, asset.getPath());
                    mimeTypes.add(mt);
                }
            }
            if (!mimeTypes.contains(assetMimeType)) {
                log.info("execute: mime type [{}] of asset [{}] is not in list of accepted mime types [" + mimeTypes
                        + "], ignoring.", assetMimeType, asset.getPath());
                return;
            }

            // creating temp directory
            tmpDir = File.createTempFile("cqdam", null);
            tmpDir.delete();
            tmpDir.mkdir();

            // make sure that tumbnails are not processed again, otherwise you
            // will end in a endless recursion
            if (null != asset) {
                // getting the resource for the node

                final Rendition original = asset.getOriginal();

                // streaming file to temp directory
                final File tmpFile = new File(tmpDir, asset.getName());
                OutputStream fos = new FileOutputStream(tmpFile);
                IOUtils.copy(original.getStream(), fos);
                IOUtils.closeQuietly(fos);

                // building command line
                CommandLine commandLine;
                String commandDescription = "";
                Map<String, String> parameters = new HashMap<String, String>();
                parameters.put("filename", tmpFile.getName());
                parameters.put("file", tmpFile.getAbsolutePath());
                parameters.put("directory", tmpDir.getAbsolutePath());
                parameters.put("basename", tmpFile.getName().replaceFirst("\\..*$", ""));
                parameters.put("extension", tmpFile.getName().replaceFirst("^.*\\.", ""));

                try {
                    for (String argument : arguments) {
                        if (argument.startsWith(Arguments.COMMANDS.getArgumentPrefix())) {
                            // Execute command line
                            final String cmd = argument.substring(Arguments.COMMANDS.getArgumentPrefix().length())
                                    .trim();
                            commandLine = CommandLine.parse(cmd, parameters);
                            commandDescription = "execute command line "+ commandLine.toString() +" for asset "+ asset.getPath() +".";
                            runCommandLineProcess(commandLine, tmpDir, commandLineProcessTimeOut, numberOfRetries, commandDescription);
                        }
                    }
                } catch (Exception e) {
                    log.error("execute: failed to "+commandDescription, e);
                }

                // go over all result files.
                for (File result : tmpDir.listFiles(new FileFilter() {
                    public boolean accept(File pathname) {
                        // ignore the original file
                        return !pathname.equals(tmpFile);
                    }
                })) {

                    // Stream output to rendition node
                    final Rendition rendition = asset.addRendition(result.getName(), new FileInputStream(result),
                        recheck(result.getName()));

                    // Extract thumbnail dimensions from args
                    final Set<ThumbnailConfig> thumbnailConfigs = new HashSet<ThumbnailConfig>();
                    for (final String str : arguments) {
                        int indexOf = str.indexOf(Arguments.THUMBNAILS.getArgumentPrefix());
                        if (indexOf > -1) {
                            final ThumbnailConfig config = CreateThumbnailProcess.parseThumbnailArguments(str.substring(indexOf
                                + Arguments.THUMBNAILS.getArgumentPrefix().length()));
                            if (null != config) {
                                thumbnailConfigs.add(config);
                                log.debug("execute: thumbnail dimensions [{}] for asset [{}].", str, asset.getPath());
                            } else {
                                log.error("execute: cannot add invalid thumbnail config [{}] for asset [{}].", str,
                                    asset.getPath());
                            }
                        }
                    }

                    List<RenditionTemplate> templates = createRenditionTemplates(rendition,
                        thumbnailConfigs.toArray(new ThumbnailConfig[] {}));
                    log.debug("thumbnail template created at [{}] with [{}] thumbnails for [" + asset.getPath() + "].",
                        rendition.getPath(), templates.size());

                    Boolean createWebRend = args.get(Arguments.GENERATE_WEB_RENDITION.name(), Boolean.class);
                    // create web-enabled rendition if flag is enabled
                    if (createWebRend != null && createWebRend) {
                        CreateWebEnabledImageProcess.Config config = webEnabledImageCreator.parseConfig(args);
                        RenditionTemplate webRendTemp = renditionMaker.createWebRenditionTemplate(rendition,
                            config.width, config.height, config.quality, config.mimeType, config.mimeTypesToKeep);
                        templates.add(webRendTemp);
                        log.debug("Web rendition template created at [{}] with [{}] thumbnails for [" + asset.getPath()
                            + "].", rendition.getPath());
                    }

                    // generate all thumbnail & web enabled renditions
                    renditionMaker.generateRenditions(asset, templates.toArray(new RenditionTemplate[] {}));

                    // if flag is enabled remove rendition generated by
                    // command.
                    Boolean delCommRend = args.get(Arguments.DELETE_COMMAND_RENDITION.name(), Boolean.class);
                    if (delCommRend != null && delCommRend) {
                        asset.removeRendition(rendition.getName());
                    }
                }
            }

        } catch (Exception e) {
            throw new WorkflowException(e);
        } finally {
            // cleaning up temp directory
            IOUtils.closeQuietly(is);
            IOUtils.closeQuietly(os);
            if (tmpDir != null) {
                try {
                    FileUtils.deleteDirectory(tmpDir);
                } catch (IOException e) {
                    throw new WorkflowException(e);
                }
            }
        }
    }

    protected void createThumbnails(final Asset asset, final Rendition rendition,
                    final Collection<ThumbnailConfig> configs) throws Exception {
        final String mimeType = rendition.getMimeType();
        final AssetHandler handler = getAssetHandler(mimeType);
        if (handler == null) {
            throw new IOException("No AssetHandler found for mimetype " + mimeType);
        }
        log.debug("createThumbnails: generating thumbnails for rendition [{}] with mime type [{}]...", asset.getPath(),
            mimeType);
        handler.createThumbnails(asset, rendition, configs);
    }

    protected String recheck(String fileName) throws RepositoryException {
        if (mimeTypeService.getMimeType(fileName.toLowerCase()) != null) {
            return mimeTypeService.getMimeType(fileName.toLowerCase());
        }
        return "application/octet-stream";
    }

    /**
    * Tries to run a command line subprocess, in case the subprocess times out retry it
    *
    * @param commandLine the command that is to be executed via commandline
    * @param binariesFolder the working directory for the commandline process
    * @param commandLineProcessTimeout the time in milliseconds after which a process will be killed
    * @param maxRetries the maximum number of time a commandline process will be retried in case of time out
    * @param commandDescription the description of the command used for logging activity
    *
    * @throws ExecuteException execution of subprocess failed or the subprocess returned a exit value indicating a failure
    * @throws CommandLineProcessTimeOutException execution of subprocess failed due to time out
    * @throws IOException
    */
    protected void runCommandLineProcess(final CommandLine commandLine, final File binariesFolder, final long commandLineProcessTimeout,
                                         final int maxRetries, final String commandDescription) throws CommandLineProcessTimeOutException, ExecuteException, IOException {
        int exitValue = 0, iteration = 0;
        DefaultExecutor exec = new DefaultExecutor();
        do {
            if (exitValue != 0) {
                log.debug("Retrying to "+commandDescription+". Retry Count = "+iteration);
            }
            exitValue = -1;
            exec.setWorkingDirectory(binariesFolder);
            log.info("execute: "+commandDescription);
            // Use ExecuteWatchDog to kill long running process
            exec.setExitValue(0);
            ExecuteWatchdog watchDog = new ExecuteWatchdog(commandLineProcessTimeout);
            exec.setWatchdog(watchDog);
            exec.setStreamHandler(new PumpStreamHandler(System.out, System.err));
            exec.setProcessDestroyer(new ShutdownHookProcessDestroyer());
            try {
                exitValue = exec.execute(commandLine);
            } catch (final ExecuteException e) {
                log.debug("execute: failed to " + commandDescription, e);
                //if process was killed by ExecuteWatchDog retry it, else throw an execute exception
                if (watchDog.killedProcess()) {
                    if (iteration < maxRetries) {
                        log.debug("Command Line process timed out after "+ commandLineProcessTimeout/1000 +" seconds and was killed by watchdog.");
                    } else {
                        throw new CommandLineProcessTimeOutException(e.getMessage(), e);
                    }
                } else {
                    throw e;
                }
            }
        } while (exec.isFailure(exitValue) && iteration++ < maxRetries);  //retry if process failed and retry remains
    }

    public String[] buildArguments(MetaDataMap metaData) {

        // the 'old' way, ensures backward compatibility
        String processArgs = metaData.get(Arguments.PROCESS_ARGS.name(), String.class);
        if (processArgs != null && !processArgs.equals("")) {
            return processArgs.split(",");
        }

        else {
            List<String> arguments = new ArrayList<String>();

            String[] commands = metaData.get(Arguments.COMMANDS.name(), String[].class);
            if (commands != null) {
                for (String command : commands) {
                    StringBuilder builder = new StringBuilder();
                    builder.append(Arguments.COMMANDS.getArgumentPrefix()).append(command);
                    arguments.add(builder.toString());
                }
            }

            String[] mimetypes = metaData.get(Arguments.MIME_TYPES.name(), String[].class);
            if (mimetypes != null) {
                for (String mimetype : mimetypes) {
                    StringBuilder builder = new StringBuilder();
                    builder.append(Arguments.MIME_TYPES.getArgumentPrefix()).append(mimetype);
                    arguments.add(builder.toString());
                }
            }

            String[] thumbnails = metaData.get(Arguments.THUMBNAILS.name(), String[].class);
            if (thumbnails != null) {
                for (String thumbnail : thumbnails) {
                    StringBuilder builder = new StringBuilder();
                    builder.append(Arguments.THUMBNAILS.getArgumentPrefix()).append(thumbnail);
                    arguments.add(builder.toString());
                }
            }

            return arguments.toArray(new String[arguments.size()]);
        }
    }

    private List<RenditionTemplate> createRenditionTemplates(Rendition rendition, ThumbnailConfig[] thumbnails) {
        List<RenditionTemplate> templates = new ArrayList<RenditionTemplate>(thumbnails.length);
        for (int i = 0; i < thumbnails.length; i++) {
            ThumbnailConfig thumb = thumbnails[i];
            templates.add(renditionMaker.createThumbnailTemplate(rendition, thumb.getWidth(), thumb.getHeight(),
                thumb.doCenter()));
        }
        return templates;
    }

}
