package com.day.cq.dam.core.process;

import com.day.cq.dam.commons.process.AbstractAssetWorkflowProcess;
import com.day.cq.workflow.PayloadMap;
import com.day.cq.workflow.WorkflowException;
import com.day.cq.workflow.WorkflowSession;
import com.day.cq.workflow.exec.WorkItem;
import com.day.cq.workflow.metadata.MetaDataMap;
import org.apache.commons.lang.StringUtils;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.ReferencePolicy;
import org.apache.felix.scr.annotations.Service;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.jcr.RepositoryException;
import javax.jcr.Session;
import java.util.ArrayList;
import java.util.List;

/**
 * The <code>DeleteDamAssetProcess</code> will delete the file in the /var/dam
 * when the asset in the /content/dam location got deleted. Deletes an
 * {@link javax.jcr.Item Item} for the Payload under the following condition:<br>
 * The Payloads relative path to a given source root exists in a given
 * destination branch.
 * <p>
 * Example with the following workflow step arguments:
 *
 * <pre>
 *    /content/dam/,
 *    /var/dam/
 * </pre>
 * <p>
 * If the Payload points to <i><b>/content/dam/</b>geometrixx/buildings</i>, The
 * Process checks if an {@link javax.jcr.Item Item} exists at
 * <i><b>/var/dam/</b>geometrixx/buildings</i>. If there is an Item and this
 * Item is not involved in a Workflowm, it will be deleted.
 * <p>
 * <b>Arguments:</b>
 * <table>
 * <thead>
 * <tr>
 * <td>Index:</td>
 * <td>Name:</td>
 * <td>Description</td>
 * <td>Example</td>
 * </tr>
 * </thead>
 * <tr>
 * <td>0</td>
 * <td>Source Root</td>
 * <td>Absolute Path the Payload's relative path should be calculated<br>
 * <b>NOTE:</b>The root must end with an <i>/</i> character</td>
 * <td>/content/dam/</td>
 * </tr>
 * <tr>
 * <td>1</td>
 * <td>Destination Root</td>
 * <td>Absolute Path the matching Item to the Payload should be search for<br>
 * <b>NOTE:</b>The root must end with an <i>/</i> character</td>
 * <td>/var/dam/</td>
 * </tr>
 * </table>
 *
 * @see AbstractAssetWorkflowProcess
 */
@Component(metatype = false)
@Service
@Property(name = "process.label", value = "Delete Dam Asset")
public class DeleteDamAssetProcess extends AbstractAssetWorkflowProcess {
    /**
     * Logger instance for this class.
     */
    private static final Logger log = LoggerFactory.getLogger(DeleteDamAssetProcess.class);

    @Reference(policy = ReferencePolicy.STATIC)
    private PayloadMap payloadMap;

    /**
     * The available arguments to this process implementation.
     */
    public enum Arguments {
        PROCESS_ARGS, SOURCE_ROOT, DESTINATION_ROOT;
    }

    public void execute(WorkItem workItem, WorkflowSession workflowSession, MetaDataMap metaData)
            throws WorkflowException {
        String[] args = buildArguments(metaData);
        if (args.length == 2) {
            String srcPrefix = args[0];
            String dstPrefix = args[1];
            try {
                final Session session = workflowSession.getSession();
                String srcPath = workItem.getWorkflowData().getPayload().toString();
                String dstPath = srcPath.replaceAll(srcPrefix, dstPrefix);

                // only delete if payload is not referenced by another process
                if (session.itemExists(dstPath) && !payloadMap.isInWorkflow(dstPath, true)) {
                    session.getItem(dstPath).remove();
                    session.save();
                }
            } catch (RepositoryException e) {
                log.error("execute: error while deleting asset; work item [{}]: ", workItem.getId(), e);
            }
        } else {
            throw new IllegalArgumentException("Insufficient arguments specified");
        }
    }

    public String[] buildArguments(MetaDataMap metaData) {

        // the 'old' way, ensures backward compatibility
        String processArgs = metaData.get(Arguments.PROCESS_ARGS.name(), String.class);
        if (processArgs != null && !processArgs.equals("")) {
            return processArgs.split(",");
        }

        else {
            List<String> arguments = new ArrayList<String>();
            String sourceRoot = metaData.get(Arguments.SOURCE_ROOT.name(), String.class);
            String destinationRoot = metaData.get(Arguments.DESTINATION_ROOT.name(), String.class);

            // the root and destination paths must end with a slash for the
            // process to work properly.
            if (StringUtils.isNotBlank(sourceRoot)) {
                arguments.add(appendSlash(sourceRoot));
            }

            if (StringUtils.isNotBlank(destinationRoot)) {
                arguments.add(appendSlash(destinationRoot));
            }

            return arguments.toArray(new String[arguments.size()]);
        }
    }

    /**
     * Makes sure the given path ends with a slash.
     *
     * @param path
     * @return
     */
    private String appendSlash(String path) {
        if (!path.endsWith("/")) {
            path = path + "/";
        }
        return path;
    }
}
