package com.day.cq.dam.core.process;

import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.day.cq.commons.Externalizer;
import com.day.cq.dam.api.DamConstants;
import com.day.cq.dam.api.jobs.AssetDownloadService;
import com.day.cq.dam.commons.process.AbstractAssetWorkflowProcess;
import com.day.cq.workflow.WorkflowException;
import com.day.cq.workflow.WorkflowSession;
import com.day.cq.workflow.exec.WorkItem;
import com.day.cq.workflow.metadata.MetaDataMap;

import org.apache.commons.lang.StringUtils;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Properties;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.ReferenceCardinality;
import org.apache.felix.scr.annotations.ReferencePolicy;
import org.apache.felix.scr.annotations.Service;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.osgi.framework.Constants;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.jcr.Session;

/**
 * The <code>DownloadAssetProcess</code> will invoke the downloadasset service to process asset(s) download.
 */
@Component(metatype = false)
@Service
@Properties({
    @Property(name = Constants.SERVICE_DESCRIPTION, value = "Download selected asset(s)."),
    @Property(name = "process.label", value = "Download Dam Asset")
})

public class DownloadAssetProcess extends AbstractAssetWorkflowProcess {
    /**
     * Logger instance for this class.
     */
    private static final Logger log = LoggerFactory.getLogger(DownloadAssetProcess.class);
    private static final String DOWNLOADASSETS = "downloadAssets";
    private static final String DOWNLOADRENDITIONS = "downloadRenditions";
    private static final String DOWNLOADSUBASSETS = "downloadSubassets";
    private static final String S7EXPORTSETTINGS = "s7exportsettings";
    private static final String EMAILTO = "emailTo";    
    private static final String CONTEXTPATH = "contextPath";

    @Reference(cardinality=ReferenceCardinality.MANDATORY_UNARY, policy = ReferencePolicy.STATIC)
    public AssetDownloadService assetDownloadService;

    public void execute(WorkItem workItem, WorkflowSession workflowSession, MetaDataMap metaData)
            throws WorkflowException {
        try {
            final Session session = workflowSession.getSession();
            final ResourceResolver rr = getResourceResolver(session);
            String payload = workItem.getWorkflowData().getPayload().toString();
            String srcPath = payload.split(".assetdownload.zip")[0];
            MetaDataMap wfMetaDataMap = workItem.getWorkflowData().getMetaDataMap();
            String contextPath = wfMetaDataMap.get(CONTEXTPATH, new String());
            // to exclude the "context-path" for "sling" to find and get the resource
            Resource resource = rr.getResource(srcPath.replace(contextPath, ""));

            // get the download job name and job options
            boolean downloadAssets = Boolean.parseBoolean(getValueFromPayload(DOWNLOADASSETS, payload));
            boolean downloadRenditions = Boolean.parseBoolean(getValueFromPayload(DOWNLOADRENDITIONS, payload));
            boolean downloadSubassets = Boolean.parseBoolean(getValueFromPayload(DOWNLOADSUBASSETS, payload));
            String s7exportsettings = getValueFromPayload(S7EXPORTSETTINGS, payload);
            if (StringUtils.isNotBlank(s7exportsettings)) {
                s7exportsettings = URLDecoder.decode(s7exportsettings, "UTF-8");
            }
            int idx = payload.indexOf("?");
            String downloadName = (idx == -1 ? "" : payload.substring(0, idx));
            downloadName = getDownloadName(downloadName);
            
            // get the download assets
            ArrayList<String> paths = getAssetPath(payload);
            Set<Resource>  downloadSet = new HashSet<Resource>(); 
            if (paths != null && paths.size() > 0) {
                for (String path: paths) {
                    // no need to decode twice since it is done in getAssetPath(...) already
                    Resource res = rr.getResource(path);
                    if (res != null) {
                        downloadSet.add(res);
                    }
                }
            } else {
                downloadSet.add(resource);
            }
            
            // process download
            String emailRecipients = wfMetaDataMap.get(EMAILTO, new String());
            String downloadUrl = assetDownloadService.assetDownload(resource, downloadSet, downloadAssets, downloadRenditions, downloadSubassets, s7exportsettings, null, null, downloadName, emailRecipients);
            if (!StringUtils.isEmpty(downloadUrl)) {
                // should check if context path exists from metaData for downloadUrl
                downloadUrl = System.getProperty("launchpad.http.server.url", getHostPrefix(rr)) + contextPath + downloadUrl;

                // store the downloadUrl as a WF metadata
                wfMetaDataMap.put(DamConstants.DOWNLOAD_URL, downloadUrl);
                
            } else{
                log.debug("downloadUrl is null or empty.");
            }
 
        } catch (Exception e) {
            log.error("execute: error while processing download asset; work item [{}]: ", workItem.getId(), e);
        }
    }
    
    private ArrayList<String> getAssetPath(String arg) {
    	ArrayList<String> paths = new ArrayList<String>();
    	
    	try {
	        int beginIdx = arg.indexOf("path=");
	        while (beginIdx != -1) {
	            int endIdx = arg.indexOf("&", beginIdx + 1);
	            endIdx = (endIdx == -1 ? arg.length() : endIdx);
	            
	            //need to pre-process this, as the '%' character gets encoded, which breaks the loop in execute()
	            String path = arg.substring(beginIdx + "path+".length(), endIdx);
	            paths.add(URLDecoder.decode(path, "UTF-8"));
	            
	            beginIdx = arg.indexOf("path=", endIdx+1);
	        }
    	} catch (Exception e) {
    		log.error("unable to parse asset paths for download: " + arg, e);
    	}
    	
    	return paths;
    }
    
    private String getValueFromPayload(String key, String argment) {
        String value = null;
        // There might be s7dam imagepreset nested command settings, 
        // So not changing argments to an array by spliting "&"  
        int beginIdx = argment.indexOf(key); 
        if (beginIdx != -1 && validKeyParameter(key, argment.substring(beginIdx - 1, beginIdx + key.length() + 1))) {
            int endIdx = argment.indexOf("&", beginIdx);
            endIdx = (endIdx == -1 ? argment.length() : endIdx);
            value = (beginIdx == -1 ? "" : argment.substring(beginIdx + (key + "=").length(), endIdx));
        }
        return value;
    }
    
    private boolean validKeyParameter(String key, String argment) {
        // we want to make sure the parameter is valid and not an injected one 
        boolean valid = false;
        List<String> params = Arrays.asList(argment.split("&|\\?|="));
        for (String param : params) {
            if (param.length() != 0) {
                valid = (param.equals(key));
            }
        }
        return valid;
    }
    
    private String getDownloadName(String pathInfo) {
        String name = pathInfo.substring(pathInfo.lastIndexOf("/") + 1, pathInfo.length());
        name = name.replaceAll("[\\[\\]\\/ :*|'\"]", ""); // remove invalid letters to JCR
        name = (name.endsWith(".zip") ? name : name + ".zip");
        return name;
    }
    private String getHostPrefix(ResourceResolver resolver) {
        Externalizer externalizer = resolver.adaptTo(Externalizer.class);
        String externalizerHost = externalizer.externalLink(resolver, Externalizer.LOCAL, "");
        if (externalizerHost != null && externalizerHost.endsWith("/")) {
            return externalizerHost.substring(0, externalizerHost.length()-1);
        } else {
            return externalizerHost;
        }
    }
}

