package com.day.cq.dam.core.process;

import java.util.List;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import com.day.cq.dam.api.DamConstants;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Service;
import org.apache.sling.api.resource.PersistenceException;
import org.apache.sling.api.resource.ResourceResolver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.cq.commons.jcr.JcrConstants;
import com.day.cq.dam.api.Asset;
import com.day.cq.dam.commons.process.AbstractAssetWorkflowProcess;
import com.day.cq.workflow.WorkflowException;
import com.day.cq.workflow.WorkflowSession;
import com.day.cq.workflow.exec.Route;
import com.day.cq.workflow.exec.WorkItem;
import com.day.cq.workflow.metadata.MetaDataMap;

/**
 * This process prevents the asset being updated after the asset got restored.
 * see bug#26535 - Revert version kicks workflow
 * @see AbstractAssetWorkflowProcess
 */
@Component(metatype = false)
@Service
@Property(name = "process.label", value = "Gate Keeper")
public class GateKeeperProcess extends AbstractAssetWorkflowProcess {
    /**
     * Logger instance for this class.
     */
    private static final Logger log = LoggerFactory.getLogger(SyncVarProcess.class);

    public void execute(WorkItem workItem, WorkflowSession workflowSession, MetaDataMap metaData)
            throws WorkflowException {
        try {
            Session session = workflowSession.getSession();
            final Asset asset = getAssetFromPayload(workItem, session);

            if (null != asset) {
                final Node assetNode = asset.adaptTo(Node.class);
                if (asset.getOriginal().getSize() == 0 || assetNode.hasProperty(JcrConstants.JCR_CONTENT + "/restored")) {
                    if (assetNode.hasProperty(JcrConstants.JCR_CONTENT + "/restored")) {
                        assetNode.getNode(JcrConstants.JCR_CONTENT).getProperty("restored").remove();
                        log.debug("It is a restored asset[{}], no processing needed. terminating workflow...", asset.getPath());
                    } else {
                        log.debug("File size is ZERO. terminating workflow...", asset.getPath());
                    }
                    workflowSession.terminateWorkflow(workItem.getWorkflow());
                    log.debug(
                        "execute: terminated workflow for asset [{}] .", asset.getPath());
                } else if (!metaData.containsKey("PROCESS_AUTO_ADVANCE") || !metaData.get("PROCESS_AUTO_ADVANCE", Boolean.class)) {
                    // if handler advance is off for this step then advance ourselves, for backwards compatibility
                    boolean advanced = false;
                    List<Route> routes = workflowSession.getRoutes(workItem);
                    for (Route route : routes) {
                        if (route.hasDefault()) {
                            workflowSession.complete(workItem, route);
                            advanced = true;
                            break;
                        }
                    }
                    // fallback if no route was marked as default
                    if (!advanced) {
                        workflowSession.complete(workItem, routes.get(0));
                    }
                    setAssetState(assetNode, session);
                } else {
                    setAssetState(assetNode, session);
                }
            } else {
                String wfPayload = workItem.getWorkflowData().getPayload().toString();
                String message = "execute: gatekeeping failed, asset [{" + wfPayload + "}] in payload doesn't exist for workflow [{" + workItem.getId() + "}].";
                throw new WorkflowException(message);
		    }
        } catch (Exception e) {
            throw new WorkflowException(e);
        }
    }

    private void setAssetState(Node assetNode, Session session) {
        try {
            if (assetNode.getNode(JcrConstants.JCR_CONTENT) != null) {
                if (!assetNode.getNode(JcrConstants.JCR_CONTENT).hasProperty(DamConstants.DAM_ASSET_STATE)) {
                    assetNode.getNode(JcrConstants.JCR_CONTENT).setProperty(DamConstants.DAM_ASSET_STATE, DamConstants.DAM_ASSET_STATE_PROCESSING);
                    session.save();
                } else {
                    if (!assetNode.getNode(JcrConstants.JCR_CONTENT).getProperty(DamConstants.DAM_ASSET_STATE).getString().equals(DamConstants.DAM_ASSET_STATE_PROCESSING)) {
                        assetNode.getNode(JcrConstants.JCR_CONTENT).setProperty(DamConstants.DAM_ASSET_STATE, DamConstants.DAM_ASSET_STATE_PROCESSING);
                        session.save();
                    }
                }
            }
        } catch (Exception e) {
            log.error("Unable to set asset state");
        }
    }
}
