package com.day.cq.dam.core.process;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.Session;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.text.StrSubstitutor;
import org.apache.commons.mail.EmailException;
import org.apache.commons.mail.HtmlEmail;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Properties;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.ReferencePolicy;
import org.apache.felix.scr.annotations.Service;
import org.apache.sling.api.resource.LoginException;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceResolverFactory;
import org.osgi.framework.Constants;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.cq.commons.Externalizer;
import com.day.cq.commons.jcr.JcrConstants;
import com.day.cq.dam.api.DamConstants;
import com.day.cq.dam.commons.process.AbstractAssetWorkflowProcess;
import com.day.cq.mailer.MessageGatewayService;
import com.day.cq.workflow.WorkflowException;
import com.day.cq.workflow.WorkflowSession;
import com.day.cq.workflow.exec.WorkItem;
import com.day.cq.workflow.metadata.MetaDataMap;

/**
 * The <code>SendDownloadAssetEmailProcess</code> will send download asset email.
 */
@Component(metatype = false)
@Service
@Properties({
    @Property(name = Constants.SERVICE_DESCRIPTION, value = "Send download asset email notification."),
    @Property(name = "process.label", value = "Send Download Asset Email Notification")
})

public class SendDownloadAssetEmailProcess extends AbstractAssetWorkflowProcess{

    private static final Logger log = LoggerFactory.getLogger(SendDownloadAssetEmailProcess.class);
    private static final String DEFAULT_CHARSET = "utf-8";
    private static final String BLANK_LINE = System.lineSeparator() + System.lineSeparator();

    /**
     * resource resolver factory.
     */
    @Reference
    private ResourceResolverFactory resourceResolverFactory;

    @Reference(policy=ReferencePolicy.STATIC)
    private MessageGatewayService messageGatewayService;
    
    public void execute(WorkItem workItem, WorkflowSession session, MetaDataMap metaData) 
            throws WorkflowException {
        MetaDataMap workflowMetaDataMap = workItem.getWorkflowData().getMetaDataMap();
        ResourceResolver resolver;
        try {
            resolver =
                resourceResolverFactory.getResourceResolver(Collections.singletonMap("user.jcr.session",
                    (Object) session.getSession()));
        } catch (final LoginException e) {
            throw new WorkflowException("could not get resource resolver", e);
        }
        
        // do not execute upon missing mail service. osgi component still needs to be available for UI
        // selection in workflow step configuration.
        if (messageGatewayService != null) { 
            try{
                ArrayList<InternetAddress> emailRecipients = getEmailRecipients(workflowMetaDataMap);
                if (workflowMetaDataMap != null) {
                	String downloadUrl = (workflowMetaDataMap.containsKey(DamConstants.DOWNLOAD_URL) ? workflowMetaDataMap.get(DamConstants.DOWNLOAD_URL).toString() : null);

                    if (emailRecipients != null && emailRecipients.size() != 0 && downloadUrl != null) {
                        String template = getEmailTemplate(metaData, session);

                        if (template != null) {
                            // get the string substitutors

                            String zipFileName = downloadUrl.substring(downloadUrl.lastIndexOf('/')+1);
                            Map<String, String> valuesMap = new HashMap<String, String>();
                            String hostPrefix = getHostPrefix(resolver);
                            valuesMap.put("host.prefix", hostPrefix);
                            valuesMap.put("zipFileName", zipFileName);
                            String encodedFileName = null;
                            try {
                                encodedFileName = URLEncoder.encode(zipFileName, "UTF-8");
                            } catch (UnsupportedEncodingException e) {
                                log.error( "Encode the file name failed before sending email in workflow", e);
                            }
                            valuesMap.put("downloadurl", downloadUrl.substring(0, downloadUrl.lastIndexOf('/')+1) + encodedFileName);
                            StrSubstitutor substitutor = new StrSubstitutor(valuesMap);
                            
                            final HtmlEmail email = createEmail(template, substitutor);
                            email.setTo( emailRecipients );
                            
                            messageGatewayService.getGateway(HtmlEmail.class).send(email);

                            log.info("Email was sent.");
                        } else {
                            log.warn("Did not send email. No email template defined");
                        }
                    } else {
                        log.warn("Did not send email. No recipient addresses or assets download URL available."); 
                    }
                } else {
                    log.warn("Did not send email. No workflow metadata is null.");
                }
            }
            catch (EmailException e) {
                e.printStackTrace();
                log.error( "Fatal error while sending email in workflow", e);
            }
        } else {
            log.warn("cannot send email, mail service unavailable. Please configure Gateway in OSGi Console");
        }
    } 
    
    private ArrayList<InternetAddress> getEmailRecipients(MetaDataMap workflowMetaDataMap) {
        ArrayList<InternetAddress> emailRecipients = null;
        String recipients = workflowMetaDataMap.get("emailTo", new String());
        try {
            recipients = URLDecoder.decode(recipients, "UTF-8");
            List<String> recipientList = Arrays.asList(recipients.split(","));
            emailRecipients = new ArrayList<InternetAddress>();

            for (String recipient: recipientList) {
                if (!StringUtils.isEmpty(recipient))
                    emailRecipients.add(new InternetAddress(recipient));
            }
        } catch (UnsupportedEncodingException e) {
            log.error("Cannot decode the recipient email address: " + e.getMessage());
        } catch (AddressException e){
            log.error("Cannot get the recipient email address: " + e.getMessage());
        }
        return emailRecipients;
    }
    
    private String getEmailTemplate(MetaDataMap workflowMetaDataMap, WorkflowSession session) {
        String template = workflowMetaDataMap.get("template", String.class);
        if (template == null) {
            // load mail template
            String templatePath = workflowMetaDataMap.get("templatePath", String.class);
            template = loadTemplate(session.getSession(), templatePath);
        }
        log.debug("Loaded template: {}", template);
        return template;
    }
    
    private HtmlEmail createEmail(final String template, final StrSubstitutor substitutor) {
        final HtmlEmail email = new HtmlEmail();
        try {
            // To handle the non-ascii chars in the subject or other headers, we can NO LONGER use InternetHeaders.
            // InternetHeaders "Read and parse the given RFC822 message... " (see the API doc).
            // And RFC822 and MIME header fields must contain only US-ASCII characters.
            // The workaround here is to use the similar technique that InternetHeaders uses,
            // which is to read the headers "till the blank line separating the header from the body. (in the API doc)"
            // We take care of the header of "subject" here since it is the only header that the current feature requires.
            String templateMessage = template.substring(0, template.indexOf(BLANK_LINE));
            List<String> templateHeaders = Arrays.asList(templateMessage.split(System.lineSeparator()));
            String subject = "";
            for (String tHeader : templateHeaders){
                if(tHeader.contains("subject")) {
                    subject = tHeader.split(":")[1].trim();
                }
            }

            // use the counting stream reader to read the mail body
            String templateBody = template.substring(template.indexOf(BLANK_LINE) + 1).trim();

            // create email
            email.setCharset(DEFAULT_CHARSET);
            
            // set subject
            log.info("Email subject: " + subject);
            if (!StringUtils.isEmpty(subject)) {
                email.setSubject(substitutor.replace(subject));
            }

            // set message body
            templateBody = substitutor.replace(templateBody);
            log.debug("Substituted mail body: {}", templateBody);
            email.setMsg(templateBody);
        } catch (Exception e){
            log.error("Create email: ", e.getMessage());
        }
        return email;
    }

    /**
     * Loads the mail templates from the repository.
     * @param path mail templates root path
     * @param session session
     * @param charSet The character set
     * @return a reader to the template or <code>null</code> if not valid.
     */
    public String loadTemplate(final Session session, final String path) {
        InputStream is = null;
        try {
            final Node content = session.getNode(path + "/" + JcrConstants.JCR_CONTENT);
            is = content.getProperty(JcrConstants.JCR_DATA).getBinary().getStream();
            final InputStreamReader r = new InputStreamReader(is, DEFAULT_CHARSET);
            final StringWriter w = new StringWriter();
            IOUtils.copy(r, w);
            return w.toString();
        } catch (final Exception e) {
            log.error("Error while loading mail template {}:{}", path, e.toString());
        } finally {
            IOUtils.closeQuietly(is);
        }
        return null;
    }

    private String getHostPrefix(ResourceResolver resolver) {
        Externalizer externalizer = resolver.adaptTo(Externalizer.class);
        String externalizerHost = externalizer.externalLink(resolver, Externalizer.LOCAL, "");
        if (externalizerHost != null && externalizerHost.endsWith("/")) {
            return externalizerHost.substring(0, externalizerHost.length()-1);
        } else {
            return externalizerHost;
        }
    }
}