/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.dam.core.process;

import static com.day.cq.commons.jcr.JcrConstants.JCR_CONTENT;
import static com.day.cq.commons.jcr.JcrConstants.JCR_DATA;
import static com.day.cq.commons.jcr.JcrConstants.NT_FILE;
import static com.day.cq.dam.api.DamConstants.ORIGINAL_FILE;

import java.io.InputStream;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Service;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.cq.dam.api.Asset;
import com.day.cq.dam.api.Rendition;
import com.day.cq.dam.commons.process.AbstractAssetWorkflowProcess;
import com.day.cq.workflow.WorkflowException;
import com.day.cq.workflow.WorkflowSession;
import com.day.cq.workflow.exec.WorkItem;
import com.day.cq.workflow.metadata.MetaDataMap;

/**
 * The <code>SyncAssetProcess</code> class will copy the (via webdav) uploaded
 * binary into the dam metadata structure.
 */
@Component(metatype = false)
@Service
@Property(name = "process.label", value = "Synchronize Asset")
public class SyncAssetProcess extends AbstractAssetWorkflowProcess {
    /**
     * Logger instance for this class.
     */
    private static final Logger log = LoggerFactory.getLogger(SyncAssetProcess.class);

    public void execute(WorkItem workItem, WorkflowSession workflowSession, MetaDataMap args) throws WorkflowException {
        try {
            final Session session = workflowSession.getSession();
            final Node binaryNode = getNodeFromPayload(workItem, session);

            if (binaryNode == null) {
                log.warn("execute: cannot sync NULL asset; work item [{}].", workItem.getId());
            } else if (!binaryNode.isNodeType(NT_FILE)) {
                log.warn("execute: cannot sync asset not of type [{}]; work item [{}].", NT_FILE, workItem.getId());
            } else if (isNotReadyForProcessing(binaryNode)) {
                // mac web dav issue
                log.debug("execute: skipping sync of file with zero size for asset [{}].", safeGetPath(binaryNode));
            } else {
                // copy asset
                log.info("execute: start syncing asset [{}]...", safeGetPath(binaryNode));
                log.debug("execute: file size for asset [{}]: [{}]", safeGetPath(binaryNode),
                        binaryNode.getProperty(JCR_CONTENT + "/" + JCR_DATA).getLength());

                final Asset asset = getAssetManager(session).createAssetForBinary(binaryNode.getPath(), false);// true);
                asset.setBatchMode(true);
                final InputStream is = binaryNode.getProperty(JCR_CONTENT + "/" + JCR_DATA).getStream();

                final Rendition original = asset.getRendition(ORIGINAL_FILE);

                final String mimeType;
                if (null != original) {
                    mimeType = original.getMimeType();
                } else {
                    mimeType = recheck(asset.getMimeType(), binaryNode);
                }

                asset.addRendition(ORIGINAL_FILE, is, mimeType);

                log.debug("execute: added original rendition to asset [{}] for binary [{}].", asset.getPath(),
                        safeGetPath(binaryNode));
            }
        } catch (RepositoryException e) {
            log.error("execute: error while syncing asset; work item [{}]: ", workItem.getId(), e);
        }
    }

}
