package com.day.cq.dam.core.process;

import com.day.cq.commons.jcr.JcrConstants;
import com.day.cq.dam.commons.process.AbstractAssetWorkflowProcess;
import com.day.cq.dam.commons.util.DamUtil;
import com.day.cq.workflow.PayloadMap;
import com.day.cq.workflow.WorkflowException;
import com.day.cq.workflow.WorkflowSession;
import com.day.cq.workflow.exec.WorkItem;
import com.day.cq.workflow.exec.Workflow;
import com.day.cq.workflow.exec.WorkflowData;
import com.day.cq.workflow.metadata.MetaDataMap;
import com.day.cq.workflow.model.WorkflowModel;
import com.day.text.Text;
import org.apache.commons.lang.StringUtils;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.ReferencePolicy;
import org.apache.felix.scr.annotations.Service;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import java.util.ArrayList;
import java.util.List;

import static com.day.cq.dam.api.DamConstants.NT_SLING_ORDEREDFOLDER;

/**
 * The <code>SyncContentProcess</code> snycs the content below /content/dam in
 * two selectable modes.
 * <p>
 * Expects its Payload to point to a <i>nt:folder</i>
 * <p>
 * <b>Arguments:</b>
 * <table>
 * <thead>
 * <tr>
 * <td>Prefix</td>
 * <td>Description</td>
 * <td>Example</td>
 * </tr>
 * </thead>
 * <tr>
 * <td>mode:</td>
 * <td>one of the follwoing:
 * <ul>
 * <li>cleanup
 * <li>sync
 * </ul>
 * The mode <i>cleanup</i> removes the {@link javax.jcr.Node Nodes} in
 * /content/dam structure that hava no counterpart in the /var structure The
 * mode <i>sync</i> starts for any {@value JcrConstants#NT_FILE} in the branch a
 * {@link com.day.cq.workflow.exec.Workflow Workflow} with the
 * {@link com.day.cq.workflow.model.WorkflowModel WorkflowModel} as given by the
 * <i>wfModelId</i> argument and the {@value JcrConstants#NT_FILE}'s path as
 * payload</td>
 * <td>mode:sync</td>
 * </tr>
 * <tr>
 * <td>wfModelId:</td>
 * <td>Idendifier of a WorkflowModel. This Workflow will be started on Assets
 * added by this Process in mode sync.<br>
 * In mode clean this argument is ignored</td>
 * <td>wfModelId:/etc/wokflow/models/syncmodell</td>
 * </tr>
 * </table>
 *
 * @see AbstractAssetWorkflowProcess
 */
@Component(metatype = false)
@Service
@Property(name = "process.label", value = "Synchronize Content")
public class SyncContentProcess extends AbstractAssetWorkflowProcess {
    /**
     * Logger instance for this class.
     */
    private static final Logger log = LoggerFactory.getLogger(SyncContentProcess.class);

    @Reference(policy = ReferencePolicy.STATIC)
    private PayloadMap payloadMap;

    /**
     * The available arguments to this process implementation.
     */
    public enum Arguments {
        PROCESS_ARGS("PROCESS_ARGS"), MODE("mode"), WF_MODEL_ID("wfModelId");
        private String argumentName;

        Arguments(String argumentName) {
            this.argumentName = argumentName;
        }

        public String getArgumentName() {
            return this.argumentName;
        }

        public String getArgumentPrefix() {
            return this.argumentName + ":";
        }
    }

    /**
     * The available modes of the {@link Arguments#MODE mode argument}.
     */
    public enum Modes {
        sync, cleanup;
    }

    public void execute(WorkItem workItem, WorkflowSession workflowSession, MetaDataMap metaData) throws WorkflowException {
        String[] args = buildArguments(metaData);
        try {
            String mode = null;
            if (getValuesFromArgs("mode", args).size() > 0) {
                mode = getValuesFromArgs("mode", args).get(0);
            }
            String wfModelId = null;
            if (getValuesFromArgs("wfModelId", args).size() > 0) {
                wfModelId = getValuesFromArgs("wfModelId", args).get(0);
            }

            final Session session = workflowSession.getSession();
            Node node = getNodeFromPayload(workItem, session);

            if (node != null && mode != null && wfModelId != null) {
                if (mode.equals("sync")) {
                    sync(node, workflowSession, wfModelId);
                }
            } else if (mode != null && mode.equals("cleanup")) {
                // is called when a folder got deleted resp. when the workflow
                // launcher
                // got notified about the deletion
                String path = workItem.getWorkflowData().getPayloadType().equals(TYPE_JCR_PATH) ? workItem
                        .getWorkflowData().getPayload().toString() : null;
                while (path != null && !session.itemExists(path)) {
                    path = Text.getRelativeParent(path, 1);
                }
                cleanup((Node) session.getItem(path));
            } else {
                log.debug("execute: referenced payload node does not exist; work item [{}].", workItem.getId());
            }
            if (session.hasPendingChanges()) {
                session.save();
            }
        } catch (RepositoryException e) {
            log.error("execute: error while syncing structure; work item [{}]: ", workItem.getId(), e);
        }
    }

    // ------------< helpers >--------------------------------------------------
    /**
     * This method synchronizes, starting from the given (/var/dam/...) node,
     * the content below /content/dam/*
     *
     * @param parentNode
     *            node below /var/dam that "contains" modifications
     * @param wfSession
     *            Workflow session
     * @param wfModelId
     *            Workflow model id. Used to start the synchronization workflow
     * @throws RepositoryException
     *             in case an error comes up while syncing
     */
    private void sync(Node parentNode, WorkflowSession wfSession, String wfModelId) throws RepositoryException {
        if (parentNode.isNodeType(JcrConstants.NT_FOLDER)) {
            checkFolder(parentNode);
            Session session = parentNode.getSession();
            NodeIterator itr = parentNode.getNodes();
            while (itr.hasNext()) {
                Node node = itr.nextNode();
                if (node.isNodeType(JcrConstants.NT_FILE)) {
                    String path = DamUtil.binaryToAssetPath(node.getPath());
                    if (!session.itemExists(path)) {
                        if (!path.contains("/._")) {
                            if (!isAlreadyInSameWorkflow(node, wfModelId)) {
                                // make sure that the asset is not already
                                // processed in the same workflow.
                                startWorkflow(node.getPath(), wfModelId, wfSession);
                                log.debug("sync: asset for [{}] does not exist, syncing with destination [{}]...",
                                        safeGetPath(node), path);
                            } else {
                                log.info("sync: skipping sync of [{}].", path);
                            }
                        } else {
                            log.debug("sync: skipping sync of [{}].", path);
                        }
                    } else {
                        log.debug("sync: asset for [{}] already exists at [{}].", safeGetPath(node), path);
                    }
                } else if (node.isNodeType(JcrConstants.NT_FOLDER)) {
                    checkFolder(node);
                    sync(node, wfSession, wfModelId);
                } else {
                    log.debug("sync: skipping sync for [{}].", safeGetPath(node));
                }
            }
        } else {
            log.debug("sync: skipping sync for [{}].", safeGetPath(parentNode));
        }
    }

    private boolean isAlreadyInSameWorkflow(Node node, String wfModelId) throws RepositoryException {
        if (payloadMap.isInWorkflow(node.getPath(), true)) {
            List<Workflow> wfs = payloadMap.getWorkflowInstances(node.getPath(), true);
            for (Workflow wf : wfs) {
                if (wf.getWorkflowModel().getId().equals(wfModelId)) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Checks if folder already exists and creates the folder if needed
     *
     * @param node
     * @throws RepositoryException
     */
    private void checkFolder(Node node) throws RepositoryException {
        String dest = getDest(node);
        if (!node.getSession().itemExists(dest)) {
            Node destNode = (Node) node.getSession().getItem(Text.getRelativeParent(dest, 1));
            destNode.addNode(node.getName(), NT_SLING_ORDEREDFOLDER);
        }
    }

    protected void startWorkflow(String assetPath, String workflowId, WorkflowSession wfSession) {
        try {
            WorkflowModel model = wfSession.getModel(workflowId);
            WorkflowData wfData = wfSession.newWorkflowData(TYPE_JCR_PATH, assetPath);
            wfSession.startWorkflow(model, wfData);
        } catch (Exception e) {
            log.warn("startWorkflow: cannot start workflow for [{}]: ", assetPath, e);
        }
    }

    private String getDest(Node node) throws RepositoryException {
        return DamUtil.binaryToAssetPath(node.getPath());
    }

    private void cleanup(Node node) throws RepositoryException {
        try {
            if (node.getSession().itemExists(getDest(node))) {
                Node contentFolder = (Node) node.getSession().getItem(getDest(node));
                log.debug("cleanup: starting cleanup for [{}]...", safeGetPath(contentFolder));
                NodeIterator itr = contentFolder.getNodes();
                while (itr.hasNext()) {
                    Node contentNode = itr.nextNode();
                    if (!node.hasNode(contentNode.getName())) {
                        // there is a "unsynchronized" folder. remove!
                        try {
                            String path = safeGetPath(contentNode);
                            contentNode.remove();
                            log.info("cleanup: removed [{}].", path);
                        } catch (RepositoryException re) {
                            log.debug("cleanup: failed to remove [{}]: ", safeGetPath(node));
                        }
                    }
                }
            }
        } catch (RepositoryException e) {
            log.debug("cleanup: failed to clean [{}]: ", safeGetPath(node));
        }
    }

    public String[] buildArguments(MetaDataMap metaData) {

        // the 'old' way, ensures backward compatibility
        String processArgs = metaData.get(Arguments.PROCESS_ARGS.name(), String.class);
        if (processArgs != null && !processArgs.equals("")) {
            return processArgs.split(",");
        }

        else {
            List<String> arguments = new ArrayList<String>();

            String currentMode = metaData.get(Arguments.MODE.name(), String.class);
            String wfModelId = metaData.get(Arguments.WF_MODEL_ID.name(), String.class);

            if (StringUtils.isNotBlank(currentMode)) {
                arguments.add(Arguments.MODE.getArgumentPrefix() + currentMode);
            }

            if (StringUtils.isNotBlank(wfModelId)) {
                arguments.add(Arguments.WF_MODEL_ID.getArgumentPrefix() + wfModelId);
            }

            return arguments.toArray(new String[arguments.size()]);
        }

    }
}
