/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.dam.core.process;

import com.day.cq.commons.jcr.JcrUtil;
import com.day.cq.dam.api.Asset;
import com.day.cq.dam.commons.process.AbstractAssetWorkflowProcess;
import com.day.cq.dam.commons.util.DamUtil;
import com.day.cq.workflow.PayloadMap;
import com.day.cq.workflow.WorkflowException;
import com.day.cq.workflow.WorkflowSession;
import com.day.cq.workflow.exec.WorkItem;
import com.day.cq.workflow.metadata.MetaDataMap;
import com.day.text.Text;
import org.apache.commons.lang.StringUtils;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.ReferencePolicy;
import org.apache.felix.scr.annotations.Service;
import org.apache.sling.api.resource.ResourceResolver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import java.util.ArrayList;
import java.util.List;

import static com.day.cq.commons.jcr.JcrConstants.JCR_CONTENT;
import static com.day.cq.commons.jcr.JcrConstants.NT_FOLDER;
import static com.day.cq.dam.api.DamConstants.NT_DAM_ASSET;
import static com.day.cq.dam.api.DamConstants.NT_SLING_ORDEREDFOLDER;
import static com.day.cq.dam.api.DamConstants.SUBASSETS_FOLDER;

/**
 * The <code>SyncContentProcess</code> syncs the content below /var/dam with
 * /content/dam in two diffrent modes.
 * <p/>
 * Process is only executed if started with a mode argument, the payload exists
 * and is currently not involved in a {@link com.day.cq.workflow.exec.Workflow
 * Workflow}
 * <p/>
 * <b>Arguments:</b>
 * <table>
 * <thead>
 * <tr>
 * <td>Prefix</td>
 * <td>Description</td>
 * <td>Example</td>
 * </tr>
 * </thead>
 * <tr>
 * <td>mode:</td>
 * <td>one of the follwoing:
 * <ul>
 * <li>cleanup
 * <li>sync
 * </ul>
 * The mode <i>cleanup</i> removes the Items deleted from the content structure.
 * <br>
 * The mode <i>sync</i> additionally adds the newly generated files to the var
 * structure.</td>
 * <td>mode:sync</td>
 * </tr>
 * </table>
 * @see AbstractAssetWorkflowProcess
 */
@Component(metatype = false)
@Service
@Property(name = "process.label", value = "Synchronize /var/dam")
public class SyncVarProcess extends AbstractAssetWorkflowProcess {
    /**
     * Logger instance for this class.
     */
    private static final Logger log = LoggerFactory.getLogger(SyncVarProcess.class);

    @Reference(policy = ReferencePolicy.STATIC)
    private PayloadMap payloadMap;

    /**
     * 200ms interval
     */
    private static final long INTERVAL = 200;

    /**
     * The available arguments to this process implementation.
     */
    public enum Arguments {
        PROCESS_ARGS("PROCESS_ARGS"), MODE("mode");
        private String argumentName;

        Arguments(String argumentName) {
            this.argumentName = argumentName;
        }

        public String getArgumentName() {
            return this.argumentName;
        }

        public String getArgumentPrefix() {
            return this.argumentName + ":";
        }
    }

    /**
     * The available modes of the {@link Arguments#MODE mode argument}.
     */
    public enum Modes {
        sync, cleanup;
    }

    public void execute(WorkItem workItem, WorkflowSession workflowSession, MetaDataMap metaData) throws WorkflowException {
        String[] args = buildArguments(metaData);

        try {
            String mode = null;
            if (getValuesFromArgs("mode", args).size() > 0) {
                mode = getValuesFromArgs("mode", args).get(0);
            }

            final Session session = workflowSession.getSession();
            Node node = getNodeFromPayload(workItem, session);

            if (node != null && mode != null) {

                if (mode.equals("sync")) {
                    sync(node, getResourceResolver(session));
                }
            } else if (mode != null && mode.equals("cleanup")) {
                // is called when a folder got deleted resp. when the workflow
                // launcher
                // got notified about the deletion
                String path = workItem.getWorkflowData().getPayloadType().equals(TYPE_JCR_PATH) ? workItem
                        .getWorkflowData().getPayload().toString() : null;
                while (path != null && !session.itemExists(path)) {
                    path = Text.getRelativeParent(path, 1);
                }
                cleanup((Node) session.getItem(path));
            } else {
                log.debug("execute: referenced payload node does not exist; work item [{}].", workItem.getId());
            }
            if (session.hasPendingChanges()) {
                session.save();
            }
        } catch (RepositoryException e) {
            log.error("execute: error while syncing structure; work item [{}]: ", workItem.getId(), e);
        }
    }

    public void execute(WorkItem workItem, WorkflowSession workflowSession) throws Exception {
        // noop
    }

    // ------------< helpers >--------------------------------------------------
    private void sync(Node node, ResourceResolver resolver) throws RepositoryException {
        Session session = node.getSession();
        if (node.isNodeType(NT_SLING_ORDEREDFOLDER)) {
            JcrUtil.createPath(getDest(node), NT_FOLDER, NT_FOLDER, session, false);
            cleanup(node);
            NodeIterator itr = node.getNodes();
            while (itr.hasNext()) {
                Node n = itr.nextNode();
                if (n.isNodeType(NT_DAM_ASSET)) {
                    log.debug("sync: syncing asset [{}]...", safeGetPath(n));
                    Asset asset = resolver.getResource(n.getPath()).adaptTo(Asset.class);
                    if (null != asset) {
                        String dest = getDest(node);
                        if (!session.itemExists(getDest(n)) && !getDest(n).contains("/._")
                                && !payloadMap.isInWorkflow(dest + "/" + asset.getName(), true)) {
                            JcrUtil.copy(asset.getOriginal().adaptTo(Node.class), (Node) session.getItem(dest),
                                    asset.getName(), true);
                        }
                    } else {
                        log.warn("sync: asset [{}] doesn't exist, cannot sync.");
                    }
                } else if (node.isNodeType(NT_SLING_ORDEREDFOLDER) && !n.getParent().isNodeType(NT_DAM_ASSET)
                        && !n.getName().equals(JCR_CONTENT)) {
                    log.debug("sync: syncing folder [{}]...", safeGetPath(n));
                    String dest = getDest(n);
                    if (!session.itemExists(dest)) {
                        Node destNode = (Node) session.getItem(getDest(node));
                        destNode.addNode(n.getName(), NT_FOLDER);
                    }
                    sync(n, resolver);
                }
            }
        } else if (node.isNodeType(NT_DAM_ASSET)) {
            Asset asset = resolver.getResource(node.getPath()).adaptTo(Asset.class);
            String dest = getDest(node.getParent());
            if (!session.itemExists(getDest(node)) && !payloadMap.isInWorkflow(dest + "/" + asset.getName(), true)
                    && !node.getParent().getName().equals(SUBASSETS_FOLDER)) {
                JcrUtil.copy(asset.getOriginal().adaptTo(Node.class), (Node) session.getItem(dest), asset.getName(),
                        true);
            }
        }
    }

    private String getDest(Node node) throws RepositoryException {
        return DamUtil.assetToBinaryPath(node.getPath());
    }

    /**
     * TODO: may check specific workflow...?
     *
     * @param node
     * @throws RepositoryException
     */
    private void cleanup(Node node) throws RepositoryException {
        try {
            if (node.getSession().itemExists(getDest(node))) {
                Node varFolder = (Node) node.getSession().getItem(getDest(node));
                log.debug("cleanup: starting cleanup for [{}]...", safeGetPath(varFolder));
                boolean isReady = false;
                long kizSize = 0;
                long count = 5;
                while (true) {
                    try {
                        Thread.sleep(INTERVAL);
                    } catch (InterruptedException e) {
                        // should not happen...
                        log.debug("cleanup: caught interrupt for [{}].", safeGetPath(varFolder));
                        break;
                    }
                    long currentSize = varFolder.getNodes().getSize();
                    isReady = (currentSize == kizSize);
                    if (isReady && count == 0) {
                        break;
                    } else if (isReady) {
                        count--;
                    } else {
                        count = 5;
                    }

                    kizSize = currentSize;
                }
                NodeIterator itr = varFolder.getNodes();
                while (itr.hasNext()) {
                    Node varNode = itr.nextNode();
                    if (!node.hasNode(varNode.getName()) && !payloadMap.isInWorkflow(varNode.getPath(), true)
                            && !varNode.getPath().contains("/._")) {
                        // there is a "unsynchronized" folder or asset. remove!
                        log.info("cleanup: removing folder [{}]...", safeGetPath(varNode));
                        try {
                            varNode.remove();
                        } catch (RepositoryException e) {
                            log.debug("cleanup: failed to remove [{}]: ", safeGetPath(node), e);
                        }
                    }
                }
            }
        } catch (RepositoryException e) {
            log.warn("cleanup: failed to clean [{}]: ", safeGetPath(node));
        }
    }

    public String[] buildArguments(MetaDataMap metaData) {

        // the 'old' way, ensures backward compatibility
        String processArgs = metaData.get(Arguments.PROCESS_ARGS.name(), String.class);
        if (processArgs != null && !processArgs.equals("")) {
            return processArgs.split(",");
        }

        else {
            List<String> arguments = new ArrayList<String>();
            String currentMode = metaData.get(Arguments.MODE.name(), String.class);
            if (StringUtils.isNotBlank(currentMode)) {
                arguments.add(Arguments.MODE.getArgumentPrefix() + currentMode);
            }
            return arguments.toArray(new String[arguments.size()]);
        }
    }

}
