/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.dam.core.process;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.Service;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.cq.dam.api.Asset;
import com.day.cq.dam.api.renditions.RenditionMaker;
import com.day.cq.dam.commons.process.AbstractAssetWorkflowProcess;
import com.day.cq.dam.core.impl.ui.preview.FolderPreviewUpdater;
import com.day.cq.workflow.WorkflowException;
import com.day.cq.workflow.WorkflowSession;
import com.day.cq.workflow.exec.WorkItem;
import com.day.cq.workflow.metadata.MetaDataMap;

/**
 * The <code>ThumbnailProcess</code> is called in a Workflow process step. This process will create one or more Thumbnails for the Asset to
 * be procesed, create web enabled image and update folder thumbnail.
 */
@Component
@Service
@Property(name = "process.label", value = "Thumbnail Process")
public class ThumbnailProcess extends AbstractAssetWorkflowProcess {

    /**
     * Logger instance for this class.
     */
    private static final Logger log = LoggerFactory.getLogger(ThumbnailProcess.class);

    @Reference
    private RenditionMaker renditionMaker;

    @Reference
    private FolderPreviewUpdater folderPreviewUpdater;

    private CreateThumbnailProcess thumbnailCreator = new CreateThumbnailProcess();

    private CreateWebEnabledImageProcess webEnabledImageCreator = new CreateWebEnabledImageProcess();

    private UpdateFolderThumbnailProcess folderThumbnailUpdater = new UpdateFolderThumbnailProcess();

    public void execute(WorkItem workItem, WorkflowSession workflowSession, MetaDataMap metaData) throws WorkflowException {

        final Asset asset = getAssetFromPayload(workItem, workflowSession.getSession());
        if (asset == null) {
            String wfPayload = workItem.getWorkflowData().getPayload().toString();
            String message = "execute: cannot process thumbnails, asset [{" + wfPayload + "}] in payload doesn't exist for workflow [{"
                + workItem.getId() + "}].";
            throw new WorkflowException(message);
        }

        final CreateThumbnailProcess.Config createThumbnailConfig = thumbnailCreator.parseConfig(metaData);
        thumbnailCreator.createThumbnails(asset, createThumbnailConfig, renditionMaker);

        final CreateWebEnabledImageProcess.Config createWebEnabledImageConfig = webEnabledImageCreator.parseConfig(metaData);
        try {
            webEnabledImageCreator.createWebEnabledImage(workItem, createWebEnabledImageConfig, asset, renditionMaker);
        } catch (RepositoryException re) {
            throw new WorkflowException(re);
        }

        Node assetNode = asset.adaptTo(Node.class);
        try {
            folderThumbnailUpdater.updateFolderThumbnail(asset, assetNode, folderPreviewUpdater);
        } catch (Exception e) {
            log.error("Error while updating folder thumbnail of asset ", asset.getPath(), e);
        }
    }
}
