/***************************************************************************/
/*                                                                         */
/*                      ADOBE CONFIDENTIAL                                 */
/*                      _ _ _ _ _ _ _ _ _ _                                */
/*                                                                         */
/*  Copyright 2014, Adobe Systems Incorporated                             */
/*  All Rights Reserved.                                                   */
/*                                                                         */
/*  NOTICE: All information contained herein is, and remains the property  */
/*  of Adobe Systems Incorporated and its suppliers, if any.  The          */
/*  intellectual and technical concepts contained herein are proprietary   */
/*  to Adobe Systems Incorporated and its suppliers and may be covered by  */
/*  U.S. and Foreign Patents, patents in process, and are protected by     */
/*  trade secret or copyright law.  Dissemination of this information or   */
/*  reproduction of this material is strictly forbidden unless prior       */
/*  written permission is obtained from Adobe Systems Incorporated.        */
/*                                                                         */
/***************************************************************************/
package com.day.cq.dam.drive.servlet;

import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.Session;
import javax.servlet.ServletException;

import org.apache.commons.collections.Predicate;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Properties;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.ReferencePolicy;
import org.apache.felix.scr.annotations.ReferencePolicyOption;
import org.apache.felix.scr.annotations.Service;
import org.apache.jackrabbit.JcrConstants;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.SlingHttpServletResponse;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.commons.json.JSONArray;
import org.apache.sling.commons.json.JSONException;
import org.apache.sling.commons.json.JSONObject;
import org.apache.sling.resource.collection.ResourceCollection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.cq.commons.servlets.AbstractPredicateServlet;
import com.day.cq.dam.api.AssetManager;
import com.day.cq.dam.api.DamConstants;
import com.day.cq.dam.drive.util.DriveRenditionPicker;
import com.day.cq.dam.drive.util.DriveUtil;
import com.day.cq.search.PredicateGroup;
import com.day.cq.search.Query;
import com.day.cq.search.QueryBuilder;

@Component(metatype = true)
@Service
@Properties({
    @org.apache.felix.scr.annotations.Property(name = "sling.servlet.resourceTypes", value = "sling/servlet/default", propertyPrivate = true),
    @Property(name = "sling.servlet.methods", value = { "GET" }),
    @Property(name = "sling.servlet.extensions", value = "json"),
    @Property(name = "sling.servlet.selectors", value = { "driveGetChildren",
        "driveGetCollection" }) })
/**
 * This servlet implements the IGetChildrenHandler of Drive API. 
 */
public class GetChildrenServlet extends AbstractPredicateServlet {

    /** logger */
    private static Logger log = LoggerFactory.getLogger(GetChildrenServlet.class);

    @Reference(policyOption = ReferencePolicyOption.GREEDY)
    DriveRenditionPicker picker;

    @Reference(policy = ReferencePolicy.STATIC)
    private QueryBuilder queryBuilder;

    /**
     * {@inheritDoc}
     */
    @Override
    protected void doGet(SlingHttpServletRequest request,
            SlingHttpServletResponse response, Predicate predicate)
            throws ServletException, IOException {
        // get the selectors array
        String[] dTypes = DriveUtil.getDataType(request);
        String selectorString = request.getRequestPathInfo().getSelectorString();
        Resource resource = request.getResource();
        ResourceResolver resolver = resource.getResourceResolver();
        JSONObject obj = null;
        if ("driveGetCollection".equals(selectorString)) {
            try {
                obj = handleCollection(resource, resolver, request, predicate,
                    dTypes);
            } catch (Exception e) {
               log.error("Error in Handling collection",e);
            }
        } else {
            try {
                obj = processFolder(request, predicate, dTypes);
            } catch (Exception e) {
                log.error("Error in Handling Folder ",e);
            }
        }
        response.addHeader("Content-Type", "application/json;charset=utf-8");
        try {
            obj.write(response.getWriter());
        } catch (JSONException e) {
            log.error("Error in returning the JSON response ",e);
        }
    }

    protected JSONObject handleCollection(Resource resource,
            ResourceResolver resolver, SlingHttpServletRequest request,
            Predicate predicate, String[] dTypes) throws Exception {
        JSONObject obj = new JSONObject();
        final AssetManager assetManager = request.getResourceResolver().adaptTo(
            AssetManager.class);
        JSONObject parentObj = new JSONObject();
        JSONObject self = new JSONObject();
        DriveUtil.fillRecipeData(request, dTypes, resource, picker,
            assetManager, self);
        JSONArray jsonArr = new JSONArray();
        Resource parent = resource.getParent();
        DriveUtil.fillRecipeData(request, dTypes, parent, picker, assetManager,
            parentObj);
        obj.put("self", self);
        obj.put("parent", parentObj);
        if (resource.adaptTo(ResourceCollection.class) == null) {
            Map<String, String> map = new HashMap<String, String>();
            map.put("path", DamConstants.MOUNTPOINT_ASSETS);
            map.put("group.0_property", "sling:resourceType");
            map.put("group.0_property.0_value", "dam/collection");
            map.put("group.0_property.1_value", "dam/smartcollection");
            // because of a bug in 5.6.1 resource type & resource super type
            // were being set on collections in reverse order
            // search for old collections with super type as dam/collection
            map.put("group.1_property", "sling:resourceSuperType");
            map.put("group.1_property.value", "dam/collection");
            map.put("group.p.or", "true");
            map.put("type", JcrConstants.NT_UNSTRUCTURED);
            map.put("p.limit", "0");
            Session session = resolver.adaptTo(Session.class);
            Query query = queryBuilder.createQuery(PredicateGroup.create(map),
                session);
            query.setHitsPerPage(0); // to return all the results
            Iterator<Resource> it = query.getResult().getResources();
            while (it.hasNext()) {
                Resource kid = it.next();
                processChildren(request, predicate, dTypes, assetManager,
                    jsonArr, kid);
            }
        } else {
            // we are browsing a particular collection
            if (!DriveUtil.isSmartCollection(resource)) {
                Resource members = resource.getChild("sling:members");
                Iterator<Resource> itr = members.getChildren().iterator();
                while (itr != null && itr.hasNext()) {
                    try {
                        Resource m = itr.next();
                        String orginalPath = m.adaptTo(Node.class).getProperty(
                            "sling:resource").getString();
                        Resource orig = resolver.getResource(orginalPath);
                        processChildren(request, predicate, dTypes,
                            assetManager, jsonArr, orig);
                    } catch (Exception e) {
                        log.warn("Error in adding a collection member", e);
                    }
                }
            }
        }
        obj.put("children", jsonArr);
        return obj;
    }

    protected JSONObject processFolder(SlingHttpServletRequest request,
            Predicate predicate, String[] dTypes) {
        Resource resource = request.getResource();
        Resource parent = resource.getParent();
        Iterator<Resource> kidz = resource.listChildren();
        final AssetManager assetManager = request.getResourceResolver().adaptTo(
            AssetManager.class);
        JSONObject obj = new JSONObject();
        JSONObject parentObj = new JSONObject();
        JSONArray jsonArr = new JSONArray();
        try {
            DriveUtil.fillRecipeData(request, dTypes, parent, picker,
                assetManager, parentObj);
            obj.put("parent", parentObj);
            JSONObject self = new JSONObject();
            DriveUtil.fillRecipeData(request, dTypes, resource, picker,
                assetManager, self);
            obj.put("self", self);
            while (kidz.hasNext()) {
                Resource kid = kidz.next();
                processChildren(request, predicate, dTypes, assetManager,
                    jsonArr, kid);
            }
            obj.put("children", jsonArr);
            return obj;

        } catch (Exception e) {
            log.error("Error in returning children", e);
        }
        return null;
    }

    protected void processChildren(SlingHttpServletRequest request,
            Predicate predicate, String[] dTypes,
            final AssetManager assetManager, JSONArray jsonArr, Resource kid) {
        try {
            if (predicate == null || predicate.evaluate(kid)) {
                JSONObject json = new JSONObject();
                DriveUtil.fillRecipeData(request, dTypes, kid, picker,
                    assetManager, json);
                jsonArr.put(json);
            }
        } catch (Exception e) {
            log.warn("Failed processing resource ", e);
        }
    }

}
