/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.day.cq.dam.handler.standard.epub;

import java.awt.Dimension;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map;

import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Service;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.cq.dam.api.Asset;
import com.day.cq.dam.api.DamConstants;
import com.day.cq.dam.api.Rendition;
import com.day.cq.dam.api.metadata.ExtractedMetadata;
import com.day.cq.dam.commons.handler.AbstractAssetHandler;
import com.day.image.Layer;

/**
 * The <code>EPubHandler</code> class ...
 */
@Component (inherit = true, metatype = false)
@Service
public class EPubHandler extends AbstractAssetHandler {

    private static final String THUMBNAIL = "thumbnail";

    /**
     * the default logger
     */
    private static final Logger log = LoggerFactory.getLogger (EPubHandler.class);

    /**
     * Mime type
     */
    public static final String EPUB_MIME_TYPE = "application/epub+zip";
    public static final String[] MIME_TYPES = new String[] {EPUB_MIME_TYPE};

    private void _setMD (Map<String, String> md, String mdname, ExtractedMetadata xmd, String xmdname) {
        String v = md.get (mdname);
        if (v != null) {
            xmd.setMetaDataProperty (xmdname, v);
        }
    }

    public ExtractedMetadata extractMetadata (final Asset asset) {
        try {
            ExtractedMetadata metadata = new ExtractedMetadata();
            log.debug ("extractMetadata: start extracting asset [{}]", asset.getPath());

            // extract metadata
            EPubExtractor mdext = new EPubExtractor ();

            InputStream is = asset.getOriginal().getStream();
            Map<String, String> md = mdext.extractMetadata (is);

            if (md == null) {
                log.debug ("unable to extract metadata");
                return null;
            }

            //_setMD (md, "date", metadata, "dc:description");
            _setMD (md, "subject", metadata, DamConstants.DC_DESCRIPTION);
            _setMD (md, "contributor", metadata, DamConstants.DC_CONTRIBUTOR);
            _setMD (md, "creator", metadata, DamConstants.DC_CREATOR);
            _setMD (md, "title", metadata, DamConstants.DC_TITLE);
            _setMD (md, "language", metadata, DamConstants.DC_LANGUAGE);
            _setMD (md, "publisher", metadata, DamConstants.DC_PUBLISHER);
            _setMD (md, "date", metadata, DamConstants.DC_DATE);
            _setMD (md, "rights", metadata, DamConstants.DC_RIGHTS);

            String coverhref = md.get ("_coverhref");
            if (coverhref != null) {
                is = asset.getOriginal().getStream();
                byte bimg[] = mdext.extractImage (is, coverhref);
                if (bimg != null) {
                    metadata.setProperty (THUMBNAIL, bimg); //this will be picked up by the getImage
                }
            }

            return metadata;
        } catch (Exception ex) {
            log.warn("extractMetadata: error while extracting metadata from EPub [{}]: ", asset.getPath(), ex);
        }
        
        return null;
    }

    public BufferedImage getImage (final Rendition rendition) throws IOException {
        return getImage (rendition, null);
    }

    public BufferedImage getImage (final Rendition rendition, final Dimension dim) throws IOException {
        Asset asset = rendition.getAsset();
        ExtractedMetadata metadata = extractMetadata (asset);
        byte buff[] = (byte[]) metadata.getProperty (THUMBNAIL);

        if (buff != null) {
            return new Layer (new ByteArrayInputStream (buff), dim).getImage();
        }
        log.warn ("Failed to retrieve thumbnail for {}", asset.getPath());
        return null;
    }

    public String[] getMimeTypes() {
        return MIME_TYPES;
    }
}
