/*
 * Copyright 1997-2011 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.dam.handler.standard.msoffice;

import com.day.cq.dam.api.Asset;
import com.day.cq.dam.api.Rendition;
import com.day.cq.dam.api.metadata.ExtractedMetadata;
import com.day.cq.dam.commons.handler.AbstractAssetHandler;
import com.day.cq.dam.handler.standard.msoffice.wmf.WmfDecoder;
import com.day.image.Layer;
import org.apache.commons.io.IOUtils;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Service;
import org.apache.poi.hpsf.Thumbnail;
import org.apache.poi.poifs.eventfilesystem.POIFSReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Toolkit;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;

import static org.apache.poi.hpsf.Thumbnail.CFTAG_MACINTOSH;
import static org.apache.poi.hpsf.Thumbnail.CFTAG_WINDOWS;


/**
 * Generic MS Office metadata handler service
 * */
@Component(inherit = true, metatype = false)
@Service
public class MSOfficeHandler extends AbstractAssetHandler {

    private static final Logger log = LoggerFactory.getLogger(MSOfficeHandler.class);

    public static final String MIMETYPE_MSWORD = "application/msword";
    public static final String MIMETYPE_MSEXCEL = "application/msexcel";
    public static final String[] MIMETYPES_SUPPORTED = new String[] {
            MIMETYPE_MSWORD,
            MIMETYPE_MSEXCEL};

    public ExtractedMetadata extractMetadata(Asset asset) {
        final MSOfficeMetadata metadata = new MSOfficeMetadata();
        final InputStream is = asset.getOriginal().getStream();
        try {
            POIFSReader r = new POIFSReader();
            // Register a listener for all documents
            MetaDataReaderListener mdrlistener = new MetaDataReaderListener();
            mdrlistener.setMetadata(metadata);
            r.registerListener(mdrlistener);
            r.read(is);
        } catch (Exception e) {
            log.debug("Failed to extract metadata", e);
        } finally {
            IOUtils.closeQuietly(is);
        }
        setMimetype(metadata, asset);
        return metadata;
    }

    @Override
    public BufferedImage getImage(final Rendition rendition) throws IOException {
        return getImage(rendition, null);
    }

    @Override
    public BufferedImage getImage(final Rendition rendition, Dimension dim) throws IOException {
        // this needs to be refactored, we should not extract meta data twice.
        final byte[] picture = (byte[]) extractMetadata(rendition.getAsset()).getProperty("picture");

        if ( picture != null ) {
            try {
                final Thumbnail thumbnail = new Thumbnail(picture);
                long cbFormat = thumbnail.getClipboardFormatTag();
                if (cbFormat == CFTAG_WINDOWS) {
                    return getWindowsImage(thumbnail, dim);
                } else if (cbFormat == CFTAG_MACINTOSH) {
                    return getMacintoshImage(thumbnail, dim);
                } else {
                    // else we do not understand the format
                    log.warn("Unsupported clipboard format {} " +
                            "see http://poi.apache.org/apidocs/org/apache/poi/hpsf/Thumbnail.html", cbFormat);
                }
            } catch (Exception e) {
                log.warn("getImage: error while getting image for MSOffice document [{}]: ", rendition.getPath(), e);
            }
        }
        return null;
    }

    /**
     * {@inheritDoc}
     */
    public String[] getMimeTypes() {
        return MIMETYPES_SUPPORTED;
    }

    /* ---------------- all private methods ---------------- */

    private BufferedImage getWindowsImage(final Thumbnail thumbnail, Dimension dim) throws Exception {
        byte[] wmfThumbnail = thumbnail.getThumbnailAsWMF();
        WmfDecoder wdec = new WmfDecoder(wmfThumbnail);
        dim = wdec.getDimension();
        BufferedImage bi = new BufferedImage(dim.width, dim.height, BufferedImage.TYPE_INT_RGB);
        Graphics2D g2 = bi.createGraphics();
        g2.drawImage(Toolkit.getDefaultToolkit().createImage(wdec), null, null);
        return new Layer(bi).getImage();
    }

    private BufferedImage getMacintoshImage(Thumbnail thumbnail, Dimension dim) throws Exception {
        // todo, transform mac thumbnail data
        throw new UnsupportedOperationException("Thumbnail has been created on Macintosh which is not yet implemented");
    }

}
