/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.dam.handler.standard.pict;

import java.awt.Rectangle;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;

import org.apache.commons.io.IOUtils;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Service;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.epfl.lse.jqd.awt.QDCanvas;
import ch.epfl.lse.jqd.basics.QDException;
import ch.epfl.lse.jqd.basics.QDPicture;

import com.day.cq.dam.api.Asset;
import com.day.cq.dam.api.Rendition;
import com.day.cq.dam.api.metadata.ExtractedMetadata;
import com.day.cq.dam.commons.handler.AbstractAssetHandler;
import com.day.cq.dam.handler.standard.helper.imageinfo.ImageInfo;

/**
 * The <code>PictHandler</code> class ...
 */
@Component(inherit = true, metatype = false)
@Service
public class PictHandler extends AbstractAssetHandler {
    /**
     * the default logger
     */
    private static final Logger log = LoggerFactory.getLogger(PictHandler.class);

    /**
     * Mime types
     */
    private static final String[] MIME_TYPES = { "image/pict", "image/x-pict" };
    
    /**
     * More Info available at www.fileformat.info/format/macpict/egff.htm
     */
    private static final int VERSION_NUMBER_2 = 0x02FF;
    
    private static final int VERSION_OPERATOR_2 = 0x0011;

    public ExtractedMetadata extractMetadata(final Asset asset) {
        ExtractedMetadata metadata = new ExtractedMetadata();
        log.debug("extractMetadata: importing asset [{}]...", asset.getPath());

        // extract metadata
        final InputStream is = asset.getOriginal().getStream();
        // get Metadata
        ImageInfo imageInfo = new ImageInfo();
        imageInfo.setDetermineImageNumber(true);
        imageInfo.setInput(is);
        imageInfo.setDetermineImageNumber(true);
        imageInfo.setCollectComments(true);
        if (imageInfo.check()) {
            metadata.setMetaDataProperty("format", imageInfo.getFormatName());
            metadata.setMetaDataProperty("MIME type", imageInfo.getMimeType());
            metadata.setMetaDataProperty("Image Width", (long) imageInfo.getWidth());
            metadata.setMetaDataProperty("Image Height", (long) imageInfo.getHeight());
            metadata.setMetaDataProperty("Bits per pixel", (long) imageInfo.getBitsPerPixel());
            metadata.setMetaDataProperty("Progressive", imageInfo.isProgressive() ? "yes" : "no");
            metadata.setMetaDataProperty("Number of images", (long) imageInfo.getNumberOfImages());
            metadata.setMetaDataProperty("Physical width (dpi)", (long) imageInfo.getPhysicalWidthDpi());
            metadata.setMetaDataProperty("Physical height (dpi)", (long) imageInfo.getPhysicalHeightDpi());
            metadata.setMetaDataProperty("Physical width (inches)", Float.toString(imageInfo.getPhysicalWidthInch()));
            metadata.setMetaDataProperty("Physical height (inches)", Float.toString(imageInfo.getPhysicalHeightInch()));
            int numComments = imageInfo.getNumberOfComments();
            metadata.setMetaDataProperty("Number of textual comments", (long) numComments);
            if (numComments > 0) {
                String comments = "";
                for (int i = 0; i < numComments; i++) {
                    comments += imageInfo.getComment(i) + "\n";
                }
                metadata.setMetaDataProperty("Comments", comments);
            }
        } else {

            byte[] imageBytes = new byte[552];

            /**
             * Reading the first 552 bytes that contain all the information
             * about image height and width.
             */

            try {
                is.read(imageBytes, 0, 551);
            } catch (IOException e) {
                log.debug("reading PICT image header failed");
            }

            /**
             * Extracting the image height and width from the image inputstream
             * according to the Macintosh PICT Header specifications.
             */
            
			int imageTopLeftXCordinate;
			int imageBottomRightXCordinate;
			int imageTopLeftYCordinate;
			int imageBottomRightYCordinate;

			// Checking for Version 2
			int versionOperator = convertToShort(imageBytes[520], imageBytes[521]);
			int versionNumber = convertToShort(imageBytes[522], imageBytes[523]);
			
			// Default Mac resolution is assumed to be at 72 Dpi.
			double defaultDpi = 72.0D;
			double imageXRatio = 1.0;
			double imageYRatio = 1.0;
			
			imageTopLeftXCordinate = convertToShort(imageBytes[512], imageBytes[513]);
			imageTopLeftYCordinate = convertToShort(imageBytes[514], imageBytes[515]);
			imageBottomRightXCordinate = convertToShort(imageBytes[516], imageBytes[517]);
			imageBottomRightYCordinate = convertToShort(imageBytes[518], imageBytes[519]);

			if (versionOperator == VERSION_OPERATOR_2 && versionNumber == VERSION_NUMBER_2) {
				int horizontalResolution = convertToShort(imageBytes[530], imageBytes[531]);
				int verticalResolution = convertToShort(imageBytes[534], imageBytes[535]);
                imageXRatio = (double) horizontalResolution/defaultDpi;
                imageYRatio = (double) verticalResolution/defaultDpi;
			} 

			long imageWidth = imageBottomRightYCordinate - imageTopLeftYCordinate;
			long imageLength = imageBottomRightXCordinate - imageTopLeftXCordinate;
			
			//Rounding off to get the whole value.
			imageWidth = Math.round(imageWidth * imageXRatio);
			imageLength = Math.round(imageLength * imageYRatio);

			if (imageWidth > 0 && imageLength > 0) {
				metadata.setMetaDataProperty("Image Width", imageWidth);
				metadata.setMetaDataProperty("Image Length", imageLength);
			}
		}

        execGenericProcessor(asset.getOriginal().getStream(), metadata);

        setMimetype(metadata, asset);
        return metadata;
    }

    public BufferedImage getImage(final Rendition rendition) throws IOException {
        final InputStream is = rendition.getStream();
        try {
            // write input stream to temp file
            File tmpFile = null;
            FileOutputStream out = null;
            try {
                tmpFile = File.createTempFile("pictin", ".pct");
                out = new FileOutputStream(tmpFile);

                try {
                    IOUtils.copy(is, out);
                } finally {
                    IOUtils.closeQuietly(out);
                }
                QDPicture pict = new QDPicture(tmpFile);
                QDCanvas canvas = new QDCanvas(pict);
                Rectangle r = canvas.getBounds();
                BufferedImage image = new BufferedImage(r.width, r.height,
                    BufferedImage.TYPE_INT_ARGB);
                canvas.paint(image.createGraphics());
                return image;

            } finally {
                IOUtils.closeQuietly(is);
                if (tmpFile != null) {
                    tmpFile.delete();
                }
            }
        } catch (IOException e) {
            log.warn(
                "getImage: error while creating an image from PICT [{}]: ",
                rendition.getPath(), e);
        } catch (QDException e) {
            log.warn(
                "getImage: error while creating an image from PICT [{}]: ",
                rendition.getPath(), e);
        }
        return null;
    }

    public String[] getMimeTypes() {
        return MIME_TYPES;
    }
    
    private int convertToShort(int byte1, int byte2) {
    	return ((byte1 & 0xFF) << 8) | (byte2 & 0xFF);
    }
}
