/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.dam.handler.standard.zip;

import com.day.cq.dam.api.Asset;
import com.day.cq.dam.api.Rendition;
import com.day.cq.dam.api.metadata.ExtractedMetadata;
import com.day.cq.dam.commons.handler.AbstractAssetHandler;
import com.day.image.Font;
import com.day.image.Layer;
import org.apache.commons.io.IOUtils;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Service;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.imageio.ImageIO;
import java.awt.Color;
import java.awt.Rectangle;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

/**
 * The <code>ZipHandler</code> class ...
 */
@Component(inherit = true, metatype = false)
@Service
public class ZipHandler extends AbstractAssetHandler {

    /**
     * the default logger
     */
    private static final Logger log = LoggerFactory.getLogger(ZipHandler.class);

    /**
     * Mime type
     */
    public static final String JAR_MIME_TYPE = "application/java-archive";
    public static final String ZIP_MIME_TYPE = "application/zip";
    public static final String[] MIME_TYPES = new String[]{JAR_MIME_TYPE, ZIP_MIME_TYPE};

    /**
     * Regular expression matching the thumbnail image file.
     */
    private static final Pattern pattern = Pattern.compile("^thumbnail\\.(gif|jpg|jpeg|png)$", Pattern.CASE_INSENSITIVE);

    public ExtractedMetadata extractMetadata(final Asset asset) {
        ExtractedMetadata metadata = new ExtractedMetadata();
        log.debug("extractMetadata: start extracting asset [{}]", asset.getPath());

        // extract metadata
        final InputStream is = asset.getOriginal().getStream();

        final ZipInputStream zis = new ZipInputStream(is, Charset.forName("Cp437"));
        final StringBuffer buffer = new StringBuffer();
        int count = 0;
        try {
            ZipEntry entry;
            while ((entry = zis.getNextEntry()) != null) {
                if (!entry.isDirectory()) {
                    buffer.append(entry.getName()).append('\n');
                    count++;
                }
                // check if file name matches thumbnail regular expression
                Matcher matcher = pattern.matcher(entry.getName());
                if (matcher.find()) {
                    try {
                        // read image from stream
                        metadata.setMetaDataProperty("Thumbnail", ImageIO.read(zis));
                    } catch (Exception e) {
                        log.debug("Error reading zip thumbnail file {}", entry.getName());
                    }
                }
                zis.closeEntry();
            }
            metadata.setMetaDataProperty("Content", buffer.toString());
            metadata.setMetaDataProperty("File Count", count);
        } catch (IOException e) {
            log.warn("extractMetadata: error while reading ZIP archive [{}]: ", asset.getPath(), e);
        } catch (IllegalArgumentException iae) {
            // bug#28534
            log.warn("extractMetadata: error while reading ZIP entry [{}]: ", asset.getPath(), iae);
        } finally {
            IOUtils.closeQuietly(zis);
        }

        // Get XMP
        execGenericProcessor(asset.getOriginal().getStream(), metadata);

        setMimetype(metadata, asset);
        return metadata;
    }

    public BufferedImage getImage(final Rendition rendition) throws IOException {
        ExtractedMetadata metadata = extractMetadata(rendition.getAsset());
        BufferedImage thumbnail = (BufferedImage) metadata.getMetaDataProperty("Thumbnail");
        String text = (String) metadata.getMetaDataProperty("Content");

        // check if there is a thumbnail image
        if (thumbnail != null) {
            return thumbnail;
        }
        // else generate a thumbnail listing the zip contents
        else if (text != null && text.length() > 0) {
            // create text layer
            final Layer layer = new Layer(500, 600, Color.WHITE);
            layer.setPaint(Color.black);
            Font font = new Font("Arial", 12);
            String displayText = getDisplayText(text, 600, 12);
            layer.drawText(10, 10, 500, 600, displayText, font, Font.ALIGN_LEFT, 0, 0);
            layer.crop(new Rectangle(510, 600));

            return layer.getImage();
        } else {
            return null;
        }
    }

    public String[] getMimeTypes() {
        return MIME_TYPES;
    }

    /**
     * This method cuts lines if the text file is too long..
     *
     * @param text       text to check
     * @param height     text box height (px)
     * @param fontheight font height (px)
     * @return the text which will fit into the box
     */
    private String getDisplayText(String text, int height, int fontheight) {
        String trimmedText = text.trim();
        int numOfLines = height / fontheight;
        String lines[] = trimmedText.split("\n");
        if (lines.length <= numOfLines) {
            return trimmedText;
        } else {
            String cutText = "";
            for (int i = 0; i < numOfLines; i++) {
                cutText += lines[i] + "\n";
            }
            return cutText;
        }
    }
}
