/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.dam.scene7.api;

import com.adobe.granite.crypto.CryptoSupport;
import com.day.cq.commons.inherit.InheritanceValueMap;


/**
 * Scene7 server access configuration.
 */
public interface S7Config {

    /**
     * Returns the configuration path of this configuration.
     * 
     * @return the s7 cloud configuration path
     */
    String getCloudConfigPath();

    /**
     * Returns the email address used for this configuration.
     * 
     * @return email address for the s7 cloud configuration
     */
    String getEmail();

    /**
     * Returns the encrypted password for the S7 service associated with this cloud configuration. To decrypt the password use the
     * {@link CryptoSupport}'s {@code unprotect} method.
     * 
     * @return encrypted password for s7 service associated with this cloud configuration
     */
    String getPassword();

    /**
     * Returns the region for this configuration.
     * 
     * @return associated s7 server region associated with this cloud configuration
     */
    String getRegion();

    /**
     * Returns the user handle associated with this configuration.
     * 
     * @return the user handle associated with the s7 server account
     */
    String getUserHandle();

    /**
     * Returns the company handle associated with this configuration.
     * 
     * @return the company handle associated with the s7 server account
     */
    String getCompanyHandle();

    /**
     * Returns the root path of the Scene7 company hierarchy (not to be confused with the actual physical path on the S7 servers).
     * 
     * @return virtual root path
     */
    String getRootPath();

    /**
     * Returns the synchronisation status as a string.
     * 
     * @return string value ("on/"off") that controls whether previously published folders/assets are kept in sync cq->s7 when moved or
     *         updated
     */
    String isSyncEnabled();

	/**
	 * Returns the publish state support status as a string.
	 * 
	 * @return string value ("on/"off") that controls whether DAM asset
	 *         activation state reflects Scene7 asset MarForPublish state
	 */
	String isPublishEnabled();

    /**
     * Returns the top most folder of the S7 company associated with this configuration.
     * 
     * @return company root folder
     */
    String getBasePath();

    /**
     * Returns a property of this configuration with the specified name.
     * 
     * @param prop
     *            the property's name
     * @return the value of the property
     */
    String get(String prop);

    /**
     * Returns the properties held in this S7 configuration object.
     * 
     * @return the properties map
     */
    InheritanceValueMap getPropertiesMap();

    /**
     * Returns the Scene7 configured publish server.
     * 
     * @return a String with the server's URL
     */
    String getPublishServer();

	/**
	 * Returns the Scene7 configured preview server.
	 * 
	 * @return a String with the server's URL
	 */
	String getPreviewServer();

    /**
     * Returns the configured Scene7 Flash Templates server.
     * 
     * @return a String with the server's URL
     */
    String getFlashTemplatesServer();

    /**
     * Returns the target path where the Scene7 assets are downloaded into CQ DAM.
     * 
     * @return a String containing the full path where the Scene7 assets will be saved into DAM
     */
    String getTargetPath();

    /**
     * Returns the default status of this configuration
     * 
     * @return boolean value indicating if this S7 configuration is the default one
     */
    boolean isDefault();

    /**
     * Returns the cached S7 FXG endpoint URL.
     * 
     * @return a String containing the FXG URL
     */
    String getFXGServer();

    /**
     * Returns the Scene7 ad-hoc folder for this configuration.
     * 
     * @return a String containing the ad-hoc folder path
     */
    String getAdhocFolder();
    
    /**
     * Returns the name of the import queue
     * 
     * @return a String containing the name of the import queue
     */
    String getImportQueueName();
    
    /**
     * Returns the topic of the associated import queue
     * 
     * @return a String containing the import queue topic
     */
    String getImportQueueTopic();


	/**
	 * Returns the Scene7 upload job parameters associated with given MIME type
	 * 
	 * @param mimeType
	 *            asset MIME type
	 * @return a String containing all Scene7 upload job parameters
	 */
	String getMimeTypeJobParams(String mimeType);
}
