/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.dam.scene7.api;

import java.util.List;
import java.util.Map;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;

/**
 * Provides access to the Scene7 cloud configuration
 */
public interface S7ConfigResolver {

    /**
     * Returns a S7Config with the corresponding parameters applied.
     * 
     * @param resourceResolver
     * @param email
     * @param password
     * @param region
     * @return the S7Config
     */
    S7Config getS7Config(ResourceResolver resourceResolver, String email,
            String password, String region);

    /**
     * Returns the S7Config generated from the configPath
     * 
     * @param resourceResolver
     *            - resource resolver to access Scene7 configuration
     * @param configPath
     *            - path to the target s7 cloud configPath settings
     * @return S7Config referenced by passed in configPath
     */
    S7Config getS7Config(ResourceResolver resourceResolver, String configPath);

    /**
     * Returns a copy of the given S7Config pointing to the shared company
     * 
     * @param resourceResolver
     *            - resource resolver to access Scene7 configuration
     * @param s7Config
     *            - config to copy from
     * @return an s7Config configured for the shared company
     */
    S7Config getSharedS7Config(ResourceResolver resourceResolver,
            S7Config s7Config);

    /**
     * Returns all the S7 configurations loaded from the children of a given
     * root path
     * 
     * @param resourceResolver
     *            - resource resolver to access Scene7 configuration
     * @param configRootPath
     *            - root path where the configurations will be loaded from
     * @return list of <code>S7Config</code> configurations
     */
    List<S7Config> getS7Configurations(ResourceResolver resourceResolver,
            String configRootPath);

    /**
     * Retrieves the Scene7 configuration associated with a DAM asset resource
     * by successively analysing the resource and then its parents. If no
     * S7Config is found this method returns null.
     * 
     * @param resource
     *            a DAM asset resource
     * @return the associated Scene7 configuration; null if no configuration is
     *         found
     */
    @Deprecated
    S7Config getS7ConfigForResource(Resource resource);

    /**
     * Retrieves the Scene7 configuration associated with an image server URL.
     * Returns null if no S7Config is found.
     * 
     * @param resource
     *            resource resolver to access Scene7 configuration
     * @param imageUrl
     *            the image server URL
     * @return return associated Scene7 configuration; null if no configuration
     *         is found
     */
    @Deprecated
    S7Config getS7ConfigForImageUrl(ResourceResolver resourceResolver,
            String imageUrl);

    /**
     * Retrieves the Scene7 configuration associated with asset by content path
     * when Scene7 metadata is not available
     *
     * @param assetPath
     *            the asset content path
     * @return associated Scene7 configuration; null if no configuration is
     *         found
     */
    S7Config getS7ConfigForAssetPath(ResourceResolver resourceResolver,
            String assetPath);

    /**
     * Retrieves the Scene7 configuration path associated with a DAM asset
     * resource by successively analysing the resource and then its parents. If
     * no S7Config is found this method returns null.
     * 
     * @param resolver
     *            resource resolver associated with user session
     * @param resource
     *            a DAM asset resource
     * @return the associated Scene7 configuration path; null if no
     *         configuration is found
     */
    String getS7ConfigPathForResource(ResourceResolver resolver,
            Resource resource);

    /**
     * Retrieves the Scene7 configuration associated with an image server URL.
     * Returns null if no S7Config is found.
     * 
     * @param resolver
     *            resource resolver associated with user session
     * @param domain
     *            delivery domain associated with configuration
     * @param companyName
     *            companyName associated with configuration
     * @return return associated Scene7 configuration path; null if no
     *         configuration is found
     */
    String getS7ConfigPathForImageUrl(ResourceResolver resolver, String domain,
            String companyName);

    /**
     * Helper method to extract domain and company information related to an
     * image url
     * 
     * @param imageUrl
     *            image url embedding the domain and company information
     * @return return a map with keys 'domain' and 'companyName' if available
     */
    Map<String, String> parseDomainAndCompanyNameForUrl(String imageUrl);

    /**
     * Retrieves the Scene7 configuration associated with an asset path. Returns
     * null if no S7Config is found.
     * 
     * @param resolver
     *            resource resolver associated with user session
     * @param assetPath
     *            asset path associated with configuration
     * @return return associated Scene7 configuration path; null if no
     *         configuration is found
     */
    String getS7ConfigPathForAssetPath(ResourceResolver resolver,
            String assetPath);

    /**
     * Scene7 region (server mapping)
     */
    public static final String REGION = "region";

    /**
     * Scene7 User email ID
     */
    public static final String EMAIL = "email";

    /**
     * Scene7 User password ID
     */
    public static final String PASSWORD = "password";

    /**
     * Scene7 generated user handle ID
     */
    public static final String USER_HANDLE = "userHandle";

    /**
     * Scene7 generated company handle ID
     */
    public static final String COMPANY_HANDLE = "companyHandle";

    /**
     * The "root path" ID in the S7 sense...see getRootPath()
     */
    public static final String ROOT_PATH = "rootPath";

    /**
     * cq root folder ID in the scene7 company folder hierarchy, is not the "root path" in the S7 (see getRootPath())
     */
    public static final String CQ_ROOT_PATH = "cqRootPath";

    /**
     * controls s7 updates and synchronization of assets/folders already managed by s7
     */
    public static final String SYNC_ENABLED = "syncEnabled";

    /**
     * controls s7 publish state updates and synchronization of assets/folders
     * already managed by s7
     */
    public static final String PUBLISH_ENABLED = "publishEnabled";

    /**
     * Default char set
     */
    public static final String CHARSET = "UTF-8";

    /**
     * The Scene7 publish server URL.
     */
    public static final String PUBLISH_SERVER = "publishServer";

    /**
     * The Scene7 preview server URL.
     */
    public static final String PREVIEW_SERVER = "previewServer";

    /**
     * The Scene7 flash templates publish server URL.
     */
    public static final String FLASH_TEMPLATES_SERVER = "flashTemplatesServer";

    /**
     * The target path property, storing the path where Scene7 assets are downloaded into CQ DAM.
     */
    public static final String TARGET_PATH = "targetPath";

    /**
     * Stores the default configuration property flag
     */
    public static final String DEFAULT_CONFIG_FLAG = "defaultConfiguration";

    /**
     * The Scene7 FXG server URL.
     */
    public static final String FXG_SERVER = "fxgServer";

    public static final String ADHOC_FOLDER = "adhocFolder";

    /**
     * The Scene7 import queue name
     */
    public static final String IMPORT_QUEUE_NAME = "importQueueName";

    /**
     * The import queue topic
     */
    public static final String IMPORT_QUEUE_TOPIC = "importQueueTopic";
}
